/***************************************************************************//**
 * @file
 * @brief USB protocol stack library API for EFM32.
 *******************************************************************************
 * # License
 * <b>Copyright 2018 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/
/** @addtogroup usb
 *@{
 */

#ifndef __EM_USB_H
#define __EM_USB_H

#include <string.h>
#include <stddef.h>

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
#if !defined(CORTEXM3_EM35X_USB)                 \
  || (EMBER_SERIAL3_MODE == EMBER_SERIAL_UNUSED) \
  || defined (USB_MSD)                           \
  || defined (USB_HID)
  #define EM_SERIAL3_ENABLED 0
#else
  #define EM_SERIAL3_ENABLED 1
#endif

#if EM_SERIAL3_ENABLED
  #include "hal/micro/cortexm3/usb/cdc/usbconfig.h"
#elif defined(USB_MSD)
  #include "hal/micro/cortexm3/usb/msd/usbconfig.h"
#elif defined(USB_HID)
  #include "hal/micro/cortexm3/usb/hid/usbconfig.h"
#else
  #include "hal/micro/cortexm3/usb/usbconfig.h"
#endif

#if defined(USB_USE_PRINTF)
#include <stdio.h>
#endif

#if USB_SELFPWRD_STATE == 0
  #define USB_SUSPEND
#endif

#ifdef __CC_ARM
#pragma anon_unions
#endif

#include <stdint.h>
#include <stdbool.h>
typedef unsigned short char16_t;
// #define false 0;
// #define true 1;

/* Debug buffer */
#if defined (USB_DEBUG)
  #include "serial/serial.h"
  #include <stdio.h>
extern char *DEBUG_BUFFER;
void USBD_PrintDebug(void);

  #define USB_PRINTDEBUG(a, b, c) DEBUG_BUFFER += sprintf(a, b, c)
#endif

/** @endcond */

/** @} END addtogroup
 */

/** @addtogroup USB_COMMON
 * @brief Sample API functions for using USB.
 *
 * See em_usb.h for source code.
 *@{
 */

/* SETUP request, direction of data stage */
#define USB_SETUP_DIR_OUT       0               /*!< Setup request data stage OUT direction value.     */
#define USB_SETUP_DIR_IN        1               /*!< Setup request data stage IN direction value.      */
#define USB_SETUP_DIR_MASK      0x80            /*!< Setup request data stage direction mask.          */
#define USB_SETUP_DIR_D2H       0x80            /*!< Setup request data stage IN direction mask.       */
#define USB_SETUP_DIR_H2D       0x00            /*!< Setup request data stage OUT direction mask.      */

/* SETUP request type */
#define USB_SETUP_TYPE_STANDARD       0         /*!< Standard setup request value.                     */
#define USB_SETUP_TYPE_CLASS          1         /*!< Class setup request value.                        */
#define USB_SETUP_TYPE_VENDOR         2         /*!< Vendor setup request value.                       */
#define USB_SETUP_TYPE_STANDARD_MASK  0x00      /*!< Standard setup request mask.                      */
#define USB_SETUP_TYPE_CLASS_MASK     0x20      /*!< Class setup request mask.                         */
#define USB_SETUP_TYPE_VENDOR_MASK    0x40      /*!< Vendor setup request mask.                        */

/* SETUP request recipient */
#define USB_SETUP_RECIPIENT_DEVICE    0         /*!< Setup request device recipient value.             */
#define USB_SETUP_RECIPIENT_INTERFACE 1         /*!< Setup request interface recipient value.          */
#define USB_SETUP_RECIPIENT_ENDPOINT  2         /*!< Setup request endpoint recipient value.           */
#define USB_SETUP_RECIPIENT_OTHER     3         /*!< Setup request other recipient value.              */

/* SETUP standard request codes for Full Speed devices */
#define GET_STATUS                0             /*!< Standard setup request GET_STATUS.                */
#define CLEAR_FEATURE             1             /*!< Standard setup request CLEAR_FEATURE.             */
#define SET_FEATURE               3             /*!< Standard setup request SET_FEATURE.               */
#define SET_ADDRESS               5             /*!< Standard setup request SET_ADDRESS.               */
#define GET_DESCRIPTOR            6             /*!< Standard setup request GET_DESCRIPTOR.            */
#define SET_DESCRIPTOR            7             /*!< Standard setup request SET_DESCRIPTOR.            */
#define GET_CONFIGURATION         8             /*!< Standard setup request GET_CONFIGURATION.         */
#define SET_CONFIGURATION         9             /*!< Standard setup request SET_CONFIGURATION.         */
#define GET_INTERFACE             10            /*!< Standard setup request GET_INTERFACE.             */
#define SET_INTERFACE             11            /*!< Standard setup request SET_INTERFACE.             */
#define SYNCH_FRAME               12            /*!< Standard setup request SYNCH_FRAME.               */

/* SETUP class request codes */
#define USB_HID_GET_REPORT        0x01          /*!< HID class setup request GET_REPORT.               */
#define USB_HID_GET_IDLE          0x02          /*!< HID class setup request GET_IDLE.                 */
#define USB_HID_SET_REPORT        0x09          /*!< HID class setup request SET_REPORT.               */
#define USB_HID_SET_IDLE          0x0A          /*!< HID class setup request SET_IDLE.                 */
#define USB_HID_SET_PROTOCOL      0x0B          /*!< HID class setup request SET_PROTOCOL.             */
#define USB_CDC_SETLINECODING     0x20          /*!< CDC class setup request SET_LINE_CODING.          */
#define USB_CDC_GETLINECODING     0x21          /*!< CDC class setup request GET_LINE_CODING.          */
#define USB_CDC_SETCTRLLINESTATE  0x22          /*!< CDC class setup request SET_CONTROL_LINE_STATE.   */
#define USB_MSD_BOTRESET          0xFF          /*!< MSD class setup request Bulk only transfer reset. */
#define USB_MSD_GETMAXLUN         0xFE          /*!< MSD class setup request Get Max LUN.              */

/* SETUP command GET/SET_DESCRIPTOR decriptor types */
#define USB_DEVICE_DESCRIPTOR             1     /*!< DEVICE descriptor value.                          */
#define USB_CONFIG_DESCRIPTOR             2     /*!< CONFIGURATION descriptor value.                   */
#define USB_STRING_DESCRIPTOR             3     /*!< STRING descriptor value.                          */
#define USB_INTERFACE_DESCRIPTOR          4     /*!< INTERFACE descriptor value.                       */
#define USB_ENDPOINT_DESCRIPTOR           5     /*!< ENDPOINT descriptor value.                        */
#define USB_DEVICE_QUALIFIER_DESCRIPTOR   6     /*!< DEVICE_QUALIFIER descriptor value.                */
#define USB_OTHER_SPEED_CONFIG_DESCRIPTOR 7     /*!< OTHER_SPEED_CONFIGURATION descriptor value.       */
#define USB_INTERFACE_POWER_DESCRIPTOR    8     /*!< INTERFACE_POWER descriptor value.                 */
#define USB_HUB_DESCRIPTOR                0x29  /*!< HUB descriptor value.                             */
#define USB_HID_DESCRIPTOR                0x21  /*!< HID descriptor value.                             */
#define USB_HID_REPORT_DESCRIPTOR         0x22  /*!< HID REPORT descriptor value.                      */
#define USB_CS_INTERFACE_DESCRIPTOR       0x24  /*!< Audio Class-specific Descriptor Type.             */

#define USB_DEVICE_DESCSIZE               18    /*!< Device descriptor size.                           */
#define USB_CONFIG_DESCSIZE               9     /*!< Configuration descriptor size.                    */
#define USB_INTERFACE_DESCSIZE            9     /*!< Interface descriptor size.                        */
#define USB_ENDPOINT_DESCSIZE             7     /*!< Endpoint descriptor size.                         */
#define USB_DEVICE_QUALIFIER_DESCSIZE     10    /*!< Device qualifier descriptor size.                 */
#define USB_OTHER_SPEED_CONFIG_DESCSIZE   9     /*!< Device other speed configuration descriptor size. */
#define USB_HID_DESCSIZE                  9     /*!< HID descriptor size.                              */
#define USB_CDC_HEADER_FND_DESCSIZE       5     /*!< CDC Header functional descriptor size.            */
#define USB_CDC_CALLMNG_FND_DESCSIZE      5     /*!< CDC Call Management functional descriptor size.   */
#define USB_CDC_ACM_FND_DESCSIZE          4     /*!< CDC Abstract Control Management functional descriptor size.*/

/* Misc. USB definitions */
#define USB_EP0_SIZE           8               /*!< The size of endpoint 0.                           */
#define USB_EP1_SIZE           8               /*!< The size of endpoint 1.                           */
#define USB_EP2_SIZE           8               /*!< The size of endpoint 2.                           */
#define USB_EP3_SIZE           64              /*!< The size of endpoint 3.                           */
#define USB_EP4_SIZE           32              /*!< The size of endpoint 4.                           */
#define USB_EP5_SIZE           64              /*!< The size of endpoint 5.                           */
#define USB_EP6_SIZE           512             /*!< The size of endpoint 6.                           */
#define USB_MAX_EP_SIZE        64               /*!< The max size of any full speed endpoint.          */
#define USB_EPTYPE_CTRL        0                /*!< Endpoint type control.                            */
#define USB_EPTYPE_ISOC        1                /*!< Endpoint type isochron.                           */
#define USB_EPTYPE_BULK        2                /*!< Endpoint type bulk.                               */
#define USB_EPTYPE_INTR        3                /*!< Endpoint type interrupt.                          */
#define USB_EP_DIR_IN          0x80             /*!< Endpoint direction mask.                          */
#define USB_SETUP_PKT_SIZE     8                /*!< Setup request packet size.                        */
#define USB_EPNUM_MASK         0x0F             /*!< Endpoint number mask.                             */
#define USB_LANGID_ENUS        0x0409           /*!< English-United States language id.                */
#define USB_MAX_DEVICE_ADDRESS 127              /*!< Maximum allowable device address.                 */

#define CONFIG_DESC_BM_REMOTEWAKEUP 0x20        /*!< Configuration descriptor attribute macro.         */
#define CONFIG_DESC_BM_SELFPOWERED  0x40        /*!< Configuration descriptor attribute macro.         */
#define CONFIG_DESC_BM_RESERVED_D7  0x80        /*!< Configuration descriptor attribute macro.         */
#define CONFIG_DESC_BM_TRANSFERTYPE 0x03        /*!< Configuration descriptor transfer type bitmask.   */
#define CONFIG_DESC_MAXPOWER_mA(x)  (((x) + 1) / 2) /*!< Configuration descriptor power macro.             */

#define DEVICE_IS_SELFPOWERED       0x0001      /*!< Standard request GET_STATUS bitmask.              */
#define REMOTE_WAKEUP_ENABLED       0x0002      /*!< Standard request GET_STATUS bitmask.              */
#define USB_FEATURE_ENDPOINT_HALT         0     /*!< Standard request CLEAR/SET_FEATURE bitmask.       */
#define USB_FEATURE_DEVICE_REMOTE_WAKEUP  1     /*!< Standard request CLEAR/SET_FEATURE bitmask.       */

#define HUB_FEATURE_PORT_RESET            4     /*!< HUB class request CLEAR/SET_PORT_FEATURE feature selector. */
#define HUB_FEATURE_PORT_POWER            8     /*!< HUB class request CLEAR/SET_PORT_FEATURE feature selector. */
#define HUB_FEATURE_C_PORT_CONNECTION     16    /*!< HUB class request CLEAR/SET_PORT_FEATURE feature selector. */
#define HUB_FEATURE_C_PORT_RESET          20    /*!< HUB class request CLEAR/SET_PORT_FEATURE feature selector. */
#define HUB_FEATURE_PORT_INDICATOR        22    /*!< HUB class request CLEAR/SET_PORT_FEATURE feature selector. */

#define USB_CLASS_CDC                     2     /*!< CDC device/interface class code.                    */
#define USB_CLASS_CDC_DATA                0x0A  /*!< CDC Data interface class code.                      */
#define USB_CLASS_CDC_ACM                 2     /*!< CDC Abstract Control Model interface subclass code. */
#define USB_CLASS_CDC_HFN                 0     /*!< CDC class Header Functional Descriptor subtype.     */
#define USB_CLASS_CDC_CMNGFN              1     /*!< CDC class Call Management Functional Descriptor subtype.*/
#define USB_CLASS_CDC_ACMFN               2     /*!< CDC class Abstract Control Management Functional Descriptor subtype.*/
#define USB_CLASS_CDC_UNIONFN             6     /*!< CDC class Union Functional Descriptor subtype.      */

#define USB_CLASS_HID                     3     /*!< HID device/interface class code.                    */
#define USB_CLASS_HID_KEYBOARD            1     /*!< HID keyboard interface protocol code.               */
#define USB_CLASS_HID_MOUSE               2     /*!< HID mouse interface protocol code.                  */

#define USB_CLASS_HUB                     9     /*!< HUB device/interface class code.                    */

#define USB_CLASS_MSD                     8     /*!< MSD device/interface class code.                    */
#define USB_CLASS_MSD_BOT_TRANSPORT       0x50  /*!< MSD Bulk Only Transport protocol.                   */
#define USB_CLASS_MSD_SCSI_CMDSET         6     /*!< MSD Subclass SCSI transparent command set.          */
#define USB_CLASS_MSD_CSW_CMDPASSED       0     /*!< MSD BOT Command status wrapper command passed code. */
#define USB_CLASS_MSD_CSW_CMDFAILED       1     /*!< MSD BOT Command status wrapper command failed code. */
#define USB_CLASS_MSD_CSW_PHASEERROR      2     /*!< MSD BOT Command status wrapper cmd phase error code.*/

#define PORT_FULL_SPEED                   1     /*!< Full speed return value for USBH_GetPortSpeed(). */
#define PORT_LOW_SPEED                    2     /*!< Low speed return value for USBH_GetPortSpeed().  */

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
/*** -------------------- Helper Macros -------------------- ***/

/** Macro for getting minimum value. */
#define EFM32_MIN(a, b)    ((a) < (b) ? (a) : (b))

/** Macro for getting maximum value. */
#define EFM32_MAX(a, b)    ((a) > (b) ? (a) : (b))

#if !defined(__GNUC__)

#define STRINGIZE(X) #X
#define EFM32_PACK_START(X) _Pragma(STRINGIZE(pack(X)))
#define EFM32_PACK_END()    _Pragma("pack()")

#ifdef __CC_ARM

/** Macros for handling aligned structs. */
#define EFM32_ALIGN(X) __align(X)
#endif
#ifdef __ICCARM__

/** Macros for handling aligned structs. */
#define EFM32_ALIGN(X) _Pragma(STRINGIZE(data_alignment = X))
#endif

#else // !defined(__GNUC__)

#define EFM32_PACK_START(x)
#define EFM32_PACK_END()
#define EFM32_ALIGN(X)

#endif // !defined(__GNUC__)

/** @endcond */

#if defined(__GNUC__)                     /* GCC compilers */
#if defined(__CHAR16_TYPE__)
typedef __CHAR16_TYPE__ char16_t;
#else
typedef unsigned short char16_t;
#endif

#elif defined(__ICCARM__)                 /* IAR compiler */
#include <uchar.h>

#elif defined(__CC_ARM)                   /* MDK-ARM compiler */
typedef unsigned short char16_t;
#endif

#define nibble2Ascii(n) ((n) + (((n) < 10) ? '0'  : 'A' - 10));

#define STATIC_CONST_STRING_DESC(_name, ...)                    \
  EFM32_PACK_START(1)                                           \
  typedef struct                                                \
  {                                                             \
    uint8_t len;                                                \
    uint8_t type;                                               \
    char16_t name[1 + sizeof((char16_t[]){ __VA_ARGS__ }) / 2]; \
  } __attribute__ ((packed)) _##_name;                          \
  EFM32_PACK_END()                                              \
  EFM32_ALIGN(4)                                                \
  EFM32_PACK_START(1)                                           \
  static const _##_name _name =                                 \
  {                                                             \
    .len  = sizeof(_##_name) - 2,                               \
    .type = USB_STRING_DESCRIPTOR,                              \
    .name = { __VA_ARGS__ },                                    \
    .name[((sizeof(_##_name) - 2) / 2) - 1] = '\0'              \
  }                                                             \
  EFM32_PACK_END()

/** Macro for creating USB compliant language string descriptors.
 *  @n Example: STATIC_CONST_STRING_DESC_LANGID( langID, 0x04, 0x09 );
 */
#define STATIC_CONST_STRING_DESC_LANGID(_name, x, y)         \
  EFM32_PACK_START(1)                                        \
  typedef struct                                             \
  {                                                          \
    uint8_t len;                                             \
    uint8_t type;                                            \
    uint8_t name[2];                                         \
  } _##_name;                                                \
  EFM32_PACK_END()                                           \
  EFM32_ALIGN(4)                                             \
  EFM32_PACK_START(1)                                        \
  static const _##_name _name __attribute__ ((aligned(4))) = \
  {                                                          \
    .len = 4,                                                \
    .type = USB_STRING_DESCRIPTOR,                           \
    .name = { y, x }                                         \
  }                                                          \
  EFM32_PACK_END()

/** Macro for creating WORD (4 byte) aligned uint8_t array with size which
 *  is a multiple of WORD size.
 *  @n Example: @n UBUF( rxBuffer, 37 );  =>  uint8_t rxBuffer[ 40 ];
 */
#if !defined(__GNUC__)
#define        UBUF(x, y) EFM32_ALIGN(4)        uint8_t x[((y) + 3) & ~3]
#define STATIC_UBUF(x, y) EFM32_ALIGN(4) static uint8_t x[((y) + 3) & ~3]
#else
#define        UBUF(x, y)        uint8_t x[((y) + 3) & ~3] __attribute__ ((aligned(4)))

/** Macro for creating WORD (4 byte) aligned static uint8_t arrays with size which
 *  is a multiple of WORD size.
 *  @n Example: @n STATIC_UBUF( rxBuffer, 37 );  =>  static uint8_t rxBuffer[ 40 ];
 */
#define STATIC_UBUF(x, y) static uint8_t x[((y) + 3) & ~3] __attribute__ ((aligned(4)))
#endif

/** @brief USB transfer status enumerator. */
typedef enum {
  /* NOTE: Please keep in sync with table errMsg[] in em_usbhal.c */
  USB_STATUS_OK              = 0,               /**< No errors detected.                               */
  USB_STATUS_REQ_ERR         = -1,              /**< Setup request error.                              */
  USB_STATUS_EP_BUSY         = -2,              /**< Endpoint is busy.                                 */
  USB_STATUS_REQ_UNHANDLED   = -3,              /**< Setup request not handled.                        */
  USB_STATUS_ILLEGAL         = -4,              /**< Illegal operation attempted.                      */
  USB_STATUS_EP_STALLED      = -5,              /**< Endpoint is stalled.                              */
  USB_STATUS_EP_ABORTED      = -6,              /**< Endpoint transfer was aborted.                    */
  USB_STATUS_EP_ERROR        = -7,              /**< Endpoint transfer error.                          */
  USB_STATUS_EP_NAK          = -8,              /**< Endpoint NAK'ed transfer request.                 */
  USB_STATUS_DEVICE_UNCONFIGURED = -9,          /**< Device is unconfigured.                           */
  USB_STATUS_DEVICE_SUSPENDED    = -10,         /**< Device is suspended.                              */
  USB_STATUS_DEVICE_RESET    = -11,             /**< Device is/was reset.                              */
  USB_STATUS_TIMEOUT         = -12,             /**< Transfer timeout.                                 */
  USB_STATUS_DEVICE_REMOVED  = -13,             /**< Device was removed.                               */
  USB_STATUS_HC_BUSY         = -14,             /**< Host channel is busy.                             */
  USB_STATUS_DEVICE_MALFUNCTION = -15,          /**< Malfunctioning device attached.                   */
  USB_STATUS_PORT_OVERCURRENT = -16,            /**< VBUS shortcircuit/overcurrent failure.            */
} USB_Status_TypeDef;

/** @} (end addtogroup USB_COMMON) */

/***************************************************************************//**
 * @addtogroup USB_DEVICE
 * @brief USB DEVICE protocol stack, see @ref usb page for detailed documentation.
 * @{
 ******************************************************************************/

/** @brief USB device state enumerator. */
typedef enum {
  USBD_STATE_NONE       = 0,                    /**< Device state is undefined/unknown.                */
  USBD_STATE_ATTACHED   = 1,                    /**< Device state is ATTACHED.                         */
  USBD_STATE_POWERED    = 2,                    /**< Device state is POWERED.                          */
  USBD_STATE_DEFAULT    = 3,                    /**< Device state is DEFAULT.                          */
  USBD_STATE_ADDRESSED  = 4,                    /**< Device state is ADDRESSED.                        */
  USBD_STATE_CONFIGURED = 4,                    /**< Device state is CONFIGURED.                       */
  USBD_STATE_SUSPENDED  = 6,                    /**< Device state is SUSPENDED.                        */
  USBD_STATE_LASTMARKER = 7,                    /**< Device state enum end marker.                     */
} USBD_State_TypeDef;

/** @} (end addtogroup USB_DEVICE) */

/** @addtogroup USB_COMMON
 *  @{*/

/** @brief USB Setup request package. */
EFM32_PACK_START(1)
typedef struct {
  union {
    struct {
      union {
        struct {
          uint8_t Recipient : 5;                /**< Request recipient (device, interface, endpoint or other).*/
          uint8_t Type      : 2;                /**< Request type (standard, class or vendor).         */
          uint8_t Direction : 1;                /**< Transfer direction of SETUP data phase.           */
        };

        uint8_t bmRequestType;                  /**< Request characteristics.                          */
      };

      uint8_t bRequest;                         /**< Request code.                                     */
      uint16_t wValue;                          /**< Varies according to request.                      */
      uint16_t wIndex;                          /**< Index or offset, varies according to request.     */
      uint16_t wLength;                         /**< Number of bytes to transfer if there is a data stage.*/
    };

    uint32_t dw[2];
  };
}  __attribute__ ((packed)) USB_Setup_TypeDef;
EFM32_PACK_END()

/** @brief USB Device Descriptor. */
EFM32_PACK_START(1)
typedef struct {
  uint8_t bLength;                              /**< Size of this descriptor in bytes                  */
  uint8_t bDescriptorType;                      /**< Constant DEVICE Descriptor Type                   */
  uint16_t bcdUSB;                              /**< USB Specification Release Number in Binary-Coded
                                                 *      Decimal                                           */
  uint8_t bDeviceClass;                         /**< Class code (assigned by the USB-IF)               */
  uint8_t bDeviceSubClass;                      /**< Subclass code (assigned by the USB-IF)            */
  uint8_t bDeviceProtocol;                      /**< Protocol code (assigned by the USB-IF)            */
  uint8_t bMaxPacketSize0;                      /**< Maximum packet size for endpoint zero             */
  uint16_t idVendor;                            /**< Vendor ID (assigned by the USB-IF)                */
  uint16_t idProduct;                           /**< Product ID (assigned by the manufacturer)         */
  uint16_t bcdDevice;                           /**< Device release number in binary-coded decimal     */
  uint8_t iManufacturer;                        /**< Index of string descriptor describing manufacturer*/
  uint8_t iProduct;                             /**< Index of string descriptor describing product     */
  uint8_t iSerialNumber;                        /**< Index of string descriptor describing the device
                                                 *      serialnumber                                      */
  uint8_t bNumConfigurations;                   /**< Number of possible configurations                 */
} __attribute__ ((packed)) USB_DeviceDescriptor_TypeDef;
EFM32_PACK_START(1)

/** @brief USB Configuration Descriptor. */
EFM32_PACK_START(1)
typedef struct {
  uint8_t bLength;                              /**< Size of this descriptor in bytes                  */
  uint8_t bDescriptorType;                      /**< Constant CONFIGURATION Descriptor Type            */
  uint16_t wTotalLength;                        /**< Total length of data returned for this
                                                 *      configuration. Includes the combined length of all
                                                 *      descriptors (configuration, interface, endpoint,
                                                 *      and class- or vendor-specific) returned for this
                                                 *      configuration.                                    */
  uint8_t bNumInterfaces;                       /**< Number of interfaces supported by this
                                                 *      configuration                                     */
  uint8_t bConfigurationValue;                  /**< Value to use as an argument to the
                                                 *      SetConfiguration request to select this
                                                 *      configuration.                                    */
  uint8_t iConfiguration;                       /**< Index of string descriptor describing this
                                                 *      configuration.                                    */
  uint8_t bmAttributes;                         /**< Configuration characteristics.
                                                 *      @n D7: Reserved (set to one)
                                                 *      @n D6: Self-powered
                                                 *      @n D5: Remote Wakeup
                                                 *      @n D4...0: Reserved (reset to zero)               */
  uint8_t bMaxPower;                            /**< Maximum power consumption of the USB device, unit
                                                 *      is 2mA per LSB                                    */
} __attribute__ ((packed)) USB_ConfigurationDescriptor_TypeDef;
EFM32_PACK_END()

/** @brief USB Interface Descriptor. */
EFM32_PACK_START(1)
typedef struct {
  uint8_t bLength;                              /**< Size of this descriptor in bytes.                 */
  uint8_t bDescriptorType;                      /**< Constant INTERFACE Descriptor Type.               */
  uint8_t bInterfaceNumber;                     /**< Number of this interface. Zero-based value
                                                 *      identifying the index in the array of concurrent
                                                 *      interfaces supported by this configuration.       */
  uint8_t bAlternateSetting;                    /**< Value used to select this alternate setting for
                                                 *       the interface identified in the prior field.      */
  uint8_t bNumEndpoints;                        /**< Number of endpoints used by this interface
                                                 *      (excluding endpoint zero). If this value is zero,
                                                 *      this interface only uses the Default Control Pipe.*/
  uint8_t bInterfaceClass;                      /**< Class code (assigned by the USB-IF). A value
                                                 *      of zero is reserved for future standardization. If
                                                 *      this field is set to FFH, the interface class is
                                                 *      vendor-specific. All other values are reserved for
                                                 *      assignment by the USB-IF.                         */
  uint8_t bInterfaceSubClass;                   /**< Subclass code (assigned by the USB-IF). These codes
                                                 *      are qualified by the value of the bInterfaceClass
                                                 *      field. If the bInterfaceClass field is reset to
                                                 *      zero, this field must also be reset to zero. If
                                                 *      the bInterfaceClass field is not set to FFH, all
                                                 *      values are reserved forassignment by the USB-IF.  */
  uint8_t bInterfaceProtocol;                   /**< Protocol code (assigned by the USB). These codes
                                                 *      are qualified by the value of the bInterfaceClass
                                                 *      and the bInterfaceSubClass fields. If an interface
                                                 *      supports class-specific requests, this code
                                                 *      identifies the protocols that the device uses as
                                                 *      defined by the specification of the device class.
                                                 *      If this field is reset to zero, the device does
                                                 *      not use a class-specific protocol on this
                                                 *      interface. If this field is set to FFH, the device
                                                 *      uses a vendor-specific protocol for this interface*/
  uint8_t iInterface;                           /**< Index of string descriptor describing this
                                                 *      interface.                                        */
} __attribute__ ((packed)) USB_InterfaceDescriptor_TypeDef;
EFM32_PACK_END()

/** @brief USB Endpoint Descriptor. */
EFM32_PACK_START(1)
typedef struct {
  uint8_t bLength;                              /**< Size of this descriptor in bytes                  */
  uint8_t bDescriptorType;                      /**< Constant ENDPOINT Descriptor Type                 */
  uint8_t bEndpointAddress;                     /**< The address of the endpoint                       */
  uint8_t bmAttributes;                         /**< This field describes the endpoint attributes      */
  uint16_t wMaxPacketSize;                      /**< Maximum packet size for the endpoint              */
  uint8_t bInterval;                            /**< Interval for polling EP for data transfers        */
} __attribute__ ((packed)) USB_EndpointDescriptor_TypeDef;
EFM32_PACK_END()

/** @brief USB String Descriptor. */
EFM32_PACK_START(1)
typedef struct {
  uint8_t len;                                  /**< Size of this descriptor in bytes.                 */
  uint8_t type;                                 /**< Constant STRING Descriptor Type.                  */
  char16_t name[];                              /**< The string encoded with UTF-16LE UNICODE charset. */
} __attribute__ ((packed)) USB_StringDescriptor_TypeDef;
EFM32_PACK_END()

/***************************************************************************//**
 * @brief
 *  USB transfer callback function.
 *
 * @details
 *  The callback function is called when a transfer has completed. An application
 *  should check the status, xferred and optionally the remaining parameters
 *  before deciding if the transfer is usable. In the case where the transfer
 *  is part of a control request data stage, the callback function should
 *  return an appropriate @ref USB_Status_TypeDef status.
 *
 * @param[in] status
 *   The transfer status. See @ref USB_Status_TypeDef.
 *
 * @param[in] xferred
 *   Number of bytes actually transferred.
 *
 * @param[in] remaining
 *   Number of bytes not transferred.
 *
 * @return
 *   @ref USB_STATUS_OK on success, else an appropriate error code.
 ******************************************************************************/
typedef int (*USB_XferCompleteCb_TypeDef)(USB_Status_TypeDef status, uint32_t xferred, uint32_t remaining);

/***************************************************************************//**
 * @brief
 *  USBTIMER callback function.
 *
 * @details
 *  The callback function is called when an USBTIMER has expired. The callback
 *  is done with interrupts disabled.
 ******************************************************************************/
typedef void (*USBTIMER_Callback_TypeDef)(void);

void  USBTIMER_DelayMs(uint32_t msec);
void  USBTIMER_DelayUs(uint32_t usec);
void  USBTIMER_Init(void);
void  USBTIMER_Start(uint32_t id, uint32_t timeout, USBTIMER_Callback_TypeDef callback);
void  USBTIMER_Stop(uint32_t id);

/** @} (end addtogroup USB_COMMON) */

#if defined(NUM_APP_TIMERS)
  #define NUM_QTIMERS (NUM_APP_TIMERS)
#else
  #define NUM_QTIMERS 0
#endif

/** @addtogroup USB_DEVICE
 *  @{*/

/*** -------------------- DEVICE mode API definitions -------------------- ***/

/***************************************************************************//**
 * @brief
 *  USB Reset callback function.
 * @details
 *  Called whenever USB reset signalling is detected on the USB port.
 ******************************************************************************/
typedef void (*USBD_UsbResetCb_TypeDef)(void);

/***************************************************************************//**
 * @brief
 *  USB Start Of Frame (SOF) interrupt callback function.
 *
 * @details
 *  Called at each SOF interrupt (if enabled),
 *
 * @param[in] sofNr
 *   Current frame number. The value rolls over to 0 after 16383 (0x3FFF).
 ******************************************************************************/
typedef void (*USBD_SofIntCb_TypeDef)(uint16_t sofNr);

/***************************************************************************//**
 * @brief
 *  USB State change callback function.
 *
 * @details
 *  Called whenever the device change state.
 *
 * @param[in] oldState
 *   The device USB state just leaved. See @ref USBD_State_TypeDef.
 *
 * @param[in] newState
 *   New (the current) USB device state. See @ref USBD_State_TypeDef.
 ******************************************************************************/
typedef void (*USBD_DeviceStateChangeCb_TypeDef)(USBD_State_TypeDef oldState, USBD_State_TypeDef newState);

/***************************************************************************//**
 * @brief
 *  USB power mode callback function.
 *
 * @details
 *  Called whenever the device stack needs to query if the device is currently
 *  self- or bus-powered. Typically when host has issued an @ref GET_STATUS
 *  setup command.
 *
 * @return
 *  True if self-powered, false otherwise.
 ******************************************************************************/
typedef bool (*USBD_IsSelfPoweredCb_TypeDef)(void);

/***************************************************************************//**
 * @brief
 *  USB setup request callback function.
 *
 * @details
 *  Called on each setup request received from host. This gives the application a
 *  possibility to extend or override standard requests, and to implement class
 *  or vendor specific requests. Return @ref USB_STATUS_OK if the request is
 *  handled, return @ref USB_STATUS_REQ_ERR if it is an illegal request or
 *  return @ref USB_STATUS_REQ_UNHANDLED to pass the request on to the default
 *  request handler.
 *
 * @param[in] setup
 *  Pointer to an USB setup packet. See @ref USB_Setup_TypeDef.
 *
 * @return
 *  An appropriate status/error code. See @ref USB_Status_TypeDef.
 ******************************************************************************/
typedef int (*USBD_SetupCmdCb_TypeDef)(const USB_Setup_TypeDef *setup);

/** @cond DO_NOT_INCLUDE_WITH_DOXYGEN */
struct USBD_Callbacks_TypeDef;
typedef struct USBD_Callbacks_TypeDef const *USBD_Callbacks_TypeDef_Pointer;

/** @endcond */

/** @brief USB Device stack initialization structure.
 *  @details This structure is passed to @ref USBD_Init() when starting up
 *  the device.                                                             */
typedef struct {
  const USB_DeviceDescriptor_TypeDef      *deviceDescriptor;      /**< Pointer to a device descriptor.                */
  const uint8_t                           *configDescriptor;      /**< Pointer to a configuration descriptor.         */
  const void * const                      *stringDescriptors;     /**< Pointer to an array of string descriptor pointers.*/
  const uint8_t numberOfStrings;                                  /**< Number of strings in string descriptor array.  */
  const uint8_t                           *bufferingMultiplier;   /**< Pointer to an array defining the size of the
                                                                   *       endpoint buffers. The size is given in
                                                                   *       multiples of endpoint size. Generally a value
                                                                   *       of 1 (single) or 2 (double) buffering should be
                                                                   *       used.                                          */
  USBD_Callbacks_TypeDef_Pointer callbacks;                       /**< Pointer to struct with callbacks
                                                                   *      (@ref USBD_Callbacks_TypeDef). These callbacks
                                                                   *      are used by the device stack to signal events
                                                                   *      to or query the application.                   */
  const uint32_t reserved;                                        /**< Reserved for future use.                       */
} USBD_Init_TypeDef;

/** @brief USB Device stack callback structure.
 *  @details Callback functions used by the device stack to signal events or
 *  query status to/from the application. See @ref USBD_Init_TypeDef. Assign
 *  members to NULL if your application don't need a specific callback. */
typedef struct USBD_Callbacks_TypeDef {
  const USBD_UsbResetCb_TypeDef usbReset;                         /**< Called whenever USB reset signalling is detected
                                                                   *      on the USB port.                                */
  const USBD_DeviceStateChangeCb_TypeDef usbStateChange;          /**< Called whenever the device change state.        */
  const USBD_SetupCmdCb_TypeDef setupCmd;                         /**< Called on each setup request received from host.*/
  const USBD_IsSelfPoweredCb_TypeDef isSelfPowered;               /**< Called whenever the device stack needs to query
                                                                   *      if the device is currently self- or bus-powered.
                                                                   *      Applies to devices which can operate in both modes.*/
  const USBD_SofIntCb_TypeDef sofInt;                             /**< Called at each SOF interrupt. If NULL, the device
                                                                   *      stack will not enable the SOF interrupt.        */
} USBD_Callbacks_TypeDef;

/*** -------------------- DEVICE mode API -------------------------------- ***/

void                USBD_AbortAllTransfers(void);
int                 USBD_AbortTransfer(int epAddr);
void                USBD_Connect(void);
void                USBD_Disconnect(void);
bool                USBD_EpIsBusy(int epAddr);
USBD_State_TypeDef  USBD_GetUsbState(void);
const char *        USBD_GetUsbStateName(USBD_State_TypeDef state);
int                 USBD_Init(const USBD_Init_TypeDef *p);
int                 USBD_Read(int epAddr, void *data, int byteCount, USB_XferCompleteCb_TypeDef callback);
int                 USBD_RemoteWakeup(void);
bool                USBD_SafeToEnterEM2(void);
int                 USBD_StallEp(int epAddr);
void                USBD_Stop(void);
int                 USBD_UnStallEp(int epAddr);
int                 USBD_Write(int epAddr, void *data, int byteCount, USB_XferCompleteCb_TypeDef callback);

/*** -------------------- UART Access -------------------------------- ***/
void usbSuspendDsr(void);

/** @} (end addtogroup USB_DEVICE) */

void usbTxData (void);
void usbForceTxData (uint8_t *data, uint8_t length);
void halInternalUart3RxIsr(uint8_t *rxData, uint8_t length, bool *rxPause);

#endif /* __EM_USB_H */
