/***************************************************************************//**
 * @brief Initialization APIs for bt stack
 *******************************************************************************
 * # License
 * <b>Copyright 2019 Silicon Laboratories Inc. www.silabs.com</b>
 *******************************************************************************
 *
 * The licensor of this software is Silicon Laboratories Inc. Your use of this
 * software is governed by the terms of Silicon Labs Master Software License
 * Agreement (MSLA) available at
 * www.silabs.com/about-us/legal/master-software-license-agreement. This
 * software is distributed to you in Source Code format and is governed by the
 * sections of the MSLA applicable to Source Code.
 *
 ******************************************************************************/


#ifndef SL_BT_STACK_INIT_H
#define SL_BT_STACK_INIT_H

#include "sl_status.h"
#include "sl_bt_api.h"
#include "sl_bt_stack_config.h"


/* Forward declarations of BGAPI class declaration structures */
struct sli_bgapi_class;
extern const struct sli_bgapi_class sli_bgapi_class_bt_dfu_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_dfu_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_system_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_system_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gap_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gap_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_advertiser_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_advertiser_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_scanner_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_scanner_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_sync_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_sync_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_connection_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_connection_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gatt_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gatt_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gatt_server_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_gatt_server_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_nvm_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_nvm_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_test_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_test_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_sm_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_sm_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_ota_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_ota_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_coex_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_coex_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_l2cap_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_l2cap_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_cte_transmitter_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_cte_transmitter_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_cte_receiver_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_cte_receiver_full;
extern const struct sli_bgapi_class sli_bgapi_class_bt_memory_profiler_optimized;
extern const struct sli_bgapi_class sli_bgapi_class_bt_memory_profiler_full;

extern void ll_initAFH();
extern void ll_initPeriodicAdv();
extern void ll_connSchAlgorithmEvenEnable();
extern void sli_bt_init_periodic_advertising();
extern void sli_bt_init_whitelisting();
extern void sl_memory_profiler_init();
extern void sli_bt_init_on_demand_start();

/* Macro to get a pointer to a BGAPI class declaration */
#ifdef SL_BT_API_FULL
#define SL_BT_BGAPI_CLASS(class_name) &sli_bgapi_class_bt_ ## class_name ## _full
#else
#define SL_BT_BGAPI_CLASS(class_name) &sli_bgapi_class_bt_ ## class_name ## _optimized
#endif

/**
 * @brief Initialize the Bluetooth stack.
 *
 * @param config The pointer of a valid configuration parameters
 * @return SL_STATUS_OK if the initialization was successful; Other error code
 *   indicates a failure.
 */
sl_status_t sl_bt_init_stack(const sl_bt_configuration_t *config);

/**
 * Initialize the Bluetooth on-demand start feature.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
static inline void sl_bt_init_on_demand_start()
{
  sli_bt_init_on_demand_start();
}

/**
 * Initialize multiprotocol. Please note that multiprotocol version of RAIL is
 * needed.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
void sl_bt_init_multiprotocol();

/**
 * Initialize and enable Adaptive Frequency Hopping (AFH) feature.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
static inline void sl_bt_init_afh()
{
  ll_initAFH();
}

/**
 * Enable even scheduling algorithm for Bluetooth
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
static inline void sl_bt_enable_even_scheduling()
{
  ll_connSchAlgorithmEvenEnable();
}

/**
 * Initialize and enable Periodic Advertising feature.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
static inline void sl_bt_init_periodic_advertising()
{
  ll_initPeriodicAdv();
  sli_bt_init_periodic_advertising();
}

/**
 * Initialize whitelisting feature.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 */
static inline void sl_bt_init_whitelisting()
{
  sli_bt_init_whitelisting();
}


/**
 * Initialize the BGAPI classes for bt stack.
 *
 * This function is automatically called by code generated by the Universal
 * Configurator if the feature is included in the project. The application is
 * not expected to call this function directly.
 *
 * @param[in] classes NULL-terminated array of pointers to class declarations.
 *   The array must remain valid until the next system reboot.
 *
 * @return SL_STATUS_OK if the initialization was successful; Other error code
 *   indicates a failure.
 */
sl_status_t sl_bt_init_classes(const struct sli_bgapi_class * const * classes);

#endif