/**
 * @file tkl_adc.c
 * @brief This is tuya tkl adc src file
 * @version 1.0
 * @date 2021-09-10
 *
 * @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
 *
 */
#include "em_cmu.h"
#include "em_iadc.h"

#include "tkl_adc.h"
#include "tkl_gpio.h"
#include "tkl_memory.h"
/******************************************************************************/
/**                  module description following                            **/
/******************************************************************************/
/* Main Features below:


*/
/******************************************************************************/
/**                  internal macro definition following                     **/
/******************************************************************************/
#define ADC_WIDTH_DEFAULT       12
#define ADC_MAX_PORT            1
#define ADC_PORT_MAX_CHANNEL    2

#define ADC0_CH0_GPIO_ID        10  //PC1
#define ADC0_CH1_GPIO_ID        17  //PD2

/******************************************************************************/
/**                  internal type definition following                      **/
/******************************************************************************/

/******************************************************************************/
/**                  internal function declaration following                 **/
/******************************************************************************/

extern UINT8_T __get_gpio_pin_num(UINT32_T pin_id);
extern UINT8_T __get_gpio_port_num(UINT32_T pin_id);


/******************************************************************************/
/**                  global veriables definition following                   **/
/******************************************************************************/
STATIC struct {
    UINT32_T gpio_id;
} sg_adc_map[ADC_PORT_MAX_CHANNEL] = {
    // default adc_port-gpio map
    {ADC0_CH0_GPIO_ID},
    {ADC0_CH1_GPIO_ID},
};

typedef struct {
    UINT8_T is_init;
    UINT8_T ch_id;
    UINT8_T width;
    TUYA_ADC_TYPE_E type;
} __ADC0_MGR_T;

STATIC __ADC0_MGR_T sg_adc0_mgr_tbl[ADC_PORT_MAX_CHANNEL];

/******************************************************************************/
/**                  functions definition following                          **/
/******************************************************************************/

STATIC UINT32_T __get_adc_gpio_from_port(TUYA_ADC_NUM_E port_num, UINT8_T ch_id) 
{
    return sg_adc_map[ch_id].gpio_id;
}

STATIC VOID_T __adc_base_init(TUYA_ADC_NUM_E port_num, __ADC0_MGR_T* cfg)
{
    IADC_Init_t v_init = IADC_INIT_DEFAULT;
    IADC_AllConfigs_t v_allconfig = IADC_ALLCONFIGS_DEFAULT;
    IADC_InitSingle_t v_initsingle = IADC_INITSINGLE_DEFAULT;
    IADC_SingleInput_t v_singleinput = IADC_SINGLEINPUT_DEFAULT;

    CMU_ClockEnable(cmuClock_IADC0, true);
    if (cfg->type == TUYA_ADC_INNER_SAMPLE_VOL) {
        v_singleinput.posInput = iadcPosInputAvdd; //iadcPosInputAvdd;
        v_singleinput.negInput = iadcNegInputGnd;
    }
    else {
        CMU_ClockEnable(cmuClock_GPIO, true);
        UINT8_T v_port, v_pin;

        UINT32_T gpio_id =__get_adc_gpio_from_port(port_num, cfg->ch_id);
        v_port = __get_gpio_port_num(gpio_id);
        v_pin = __get_gpio_pin_num(gpio_id);

        v_singleinput.posInput =(IADC_PosInput_t)(iadcPosInputPortAPin0 + (v_port - GPIO_PORTA) * 16 + v_pin);
        v_singleinput.negInput = iadcNegInputGnd;
        v_allconfig.configs[0].analogGain = iadcCfgAnalogGain0P5x;

        GPIO_PinModeSet((GPIO_Port_TypeDef)v_port, v_pin, gpioModeDisabled, 0);
        if (v_port == GPIO_PORTA) {
            GPIO->ABUSALLOC |= (GPIO_ABUSALLOC_AEVEN0_ADC0 | GPIO_ABUSALLOC_AODD0_ADC0);
        } 
        else if (v_port == GPIO_PORTB) {
            GPIO->BBUSALLOC |= (GPIO_BBUSALLOC_BEVEN0_ADC0 | GPIO_BBUSALLOC_BODD0_ADC0);
        }
        else {
            GPIO->CDBUSALLOC |= (GPIO_CDBUSALLOC_CDEVEN0_ADC0 | GPIO_CDBUSALLOC_CDODD0_ADC0);
        }
    }

    CMU_ClockEnable(cmuClock_IADC0, true);
    CMU_ClockSelectSet(cmuClock_IADCCLK, cmuSelect_FSRCO);

    IADC_init(IADC0, &v_init, &v_allconfig);
    IADC_initSingle(IADC0, &v_initsingle, &v_singleinput);

    // When selecting SUPPLY for PORTPOS and GND for PORTNEG,
    // PINNEG should be configured for an odd number (1, 3, 5...) to avoid a
    // polarity error.
    IADC0->SINGLE |= 1;
}

OPERATE_RET tkl_adc_mapping_to_gpio(TUYA_ADC_NUM_E port_num, UINT8_T ch_id, UINT32_T gpio_id)
{
    if (port_num >= ADC_MAX_PORT || ch_id >= ADC_PORT_MAX_CHANNEL) {
        return OPRT_INVALID_PARM;
    }

    return sg_adc_map[ch_id].gpio_id;
}

OPERATE_RET tkl_adc_init(TUYA_ADC_NUM_E port_num, TUYA_ADC_BASE_CFG_T *cfg)
{
    UINT8_T i = 0;
    UINT8_T ch_index = 0;
    UINT32_T value = 1;

    if (port_num >= ADC_MAX_PORT || cfg == NULL || cfg->ch_nums !=0) {
        return OPRT_INVALID_PARM;
    }
    if (cfg->ch_nums > ADC_PORT_MAX_CHANNEL) {
        return OPRT_INVALID_PARM;
    }
    for (; ch_index < 32; ch_index++)
    {
        if(cfg->ch_list.data & value){
            break;
        }

        cfg->ch_list.data >>=1;
    }
    
    for(i = 0; i < cfg->ch_nums; i++) {
        sg_adc0_mgr_tbl[i].ch_id = ch_index;
        sg_adc0_mgr_tbl[i].type = cfg->type;
        // NOTE: ADC width is a fixed value:12
        sg_adc0_mgr_tbl[i].width = ADC_WIDTH_DEFAULT;
        sg_adc0_mgr_tbl[i].is_init = 1;
    }

    return OPRT_OK; 
}

VOID_T __adc_external_sample_gpio_deinit(TUYA_ADC_NUM_E port_num, UINT8_T ch_id)
{
    UINT8_T v_port, v_pin;
    UINT32_T gpio_id = __get_adc_gpio_from_port(port_num, ch_id);
    v_port = __get_gpio_port_num(gpio_id);
    v_pin = __get_gpio_pin_num(gpio_id);

    CMU_ClockEnable(cmuClock_GPIO, false);
    GPIO_PinModeSet((GPIO_Port_TypeDef)v_port, v_pin, gpioModeDisabled, 0);
    if (v_port == GPIO_PORTA) {
        GPIO->ABUSALLOC &= ~(GPIO_ABUSALLOC_AEVEN0_ADC0 | GPIO_ABUSALLOC_AODD0_ADC0);
    }
    else if (v_port == GPIO_PORTB) {
        GPIO->BBUSALLOC &= ~(GPIO_BBUSALLOC_BEVEN0_ADC0 | GPIO_BBUSALLOC_BODD0_ADC0);
    }
    else {
        GPIO->CDBUSALLOC &= ~(GPIO_CDBUSALLOC_CDEVEN0_ADC0 | GPIO_CDBUSALLOC_CDODD0_ADC0);
    }
}

OPERATE_RET tkl_adc_deinit(TUYA_ADC_NUM_E port_num)
{
    UINT8_T i;
    if (port_num >= ADC_MAX_PORT) {
        return OPRT_INVALID_PARM;
    }

    IADC_reset(IADC0);
    CMU_ClockEnable(cmuClock_IADC0, false);

    for (i = 0; i < ADC_PORT_MAX_CHANNEL; i++) {
        if (!sg_adc0_mgr_tbl[i].is_init) {
            continue;
        }
        if (sg_adc0_mgr_tbl[i].type == TUYA_ADC_EXTERNAL_SAMPLE_VOL){
            __adc_external_sample_gpio_deinit(port_num, sg_adc0_mgr_tbl[i].ch_id);
        }
        sg_adc0_mgr_tbl[i].is_init = 0;
    }

    return OPRT_OK;
}

OPERATE_RET tkl_adc_read_data(TUYA_ADC_NUM_E port_num, INT32_T *buff, UINT16_T len)
{
    return OPRT_NOT_SUPPORTED;
}

OPERATE_RET tkl_adc_read_single_channel(TUYA_ADC_NUM_E port_num, UINT8_T ch_id, INT32_T *data)
{
    if (data == NULL || 
        port_num >= ADC_MAX_PORT || 
        ch_id >= ADC_PORT_MAX_CHANNEL) {
        return OPRT_INVALID_PARM;
    }

    if (!sg_adc0_mgr_tbl[ch_id].is_init) {
        return OPRT_INVALID_PARM;
    }

    IADC_reset(IADC0);
    __adc_base_init(port_num, &sg_adc0_mgr_tbl[ch_id]);

    IADC_Result_t v_result;
    IADC_command(IADC0, iadcCmdStartSingle);
    while (!(IADC0->STATUS & (_IADC_STATUS_CONVERTING_MASK | _IADC_STATUS_SINGLEFIFODV_MASK)));
    
    v_result = IADC_pullSingleFifoResult(IADC0);
    if (v_result.data == 0) {
        IADC_command(IADC0, iadcCmdStartSingle);
        while (!(IADC0->STATUS & (_IADC_STATUS_CONVERTING_MASK | _IADC_STATUS_SINGLEFIFODV_MASK)));
        v_result = IADC_pullSingleFifoResult(IADC0);
    }

    IADC_reset(IADC0);
    CMU_ClockEnable(cmuClock_IADC0, false);
    
    if(sg_adc0_mgr_tbl[ch_id].type == TUYA_ADC_INNER_SAMPLE_VOL) {
        *data = (INT32_T)(v_result.data*4);
    }
    else {
        *data = (INT32_T)(v_result.data*2);
        __adc_external_sample_gpio_deinit(port_num, ch_id);
    }
    
    return OPRT_OK;
}


UINT8_T tkl_adc_width_get(TUYA_ADC_NUM_E port_num)
{
    return ADC_WIDTH_DEFAULT;
}


UINT32_T tkl_adc_ref_voltage_get(TUYA_ADC_NUM_E port_num)
{
    return 1210;
}


INT32_T tkl_adc_temperature_get(VOID)
{
    return 0;
}

OPERATE_RET tkl_adc_read_voltage(TUYA_ADC_NUM_E port_num, INT32_T *buff, UINT16_T len)
{
    // tkl_adc_read_single_channel(port_num, ch_id, &reg_data);
    // tkl_adc_ref_voltage_get()*reg_data / 4095;
    return OPRT_NOT_SUPPORTED;
}

/******************************************************************************/
/**                              EOF                                         **/
/******************************************************************************/
