/**
 * @file tkl_flash.c
 * @brief This is tuya tkl flash src file
 * @version 1.0
 * @date 2021-09-10
 *
 * @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
 *
 */
#include "em_cmu.h"
#include "em_se.h"
#include "em_msc.h"

#include "tkl_memory.h"
#include "tkl_flash.h"
/******************************************************************************/
/**                  module description following                            **/
/******************************************************************************/
/* Main Features below:
                          MG21 1M flash map new OTA 
|**** 0x00000000 ****|**** 32K *****|**** 0x00008000 *****|***** bootloader******|
|**** 0x00008000 ****|**** 496K ****|**** 0x00084000 *****|***** app code********|
|**** 0x00084000 ****|**** 16K *****|**** 0x00088000 *****|*** diff backup*******|
|**** 0x00088000 ****|**** 352K ****|**** 0x000E0000 *****|***** ota*************|// diff ota size = ota; full ota size = ota+diff mange;
|**** 0x000E0000 ****|**** 16K *****|**** 0x000E4000 *****|**** diff mange*******|
|**** 0x000E4000 ****|**** 16K *****|**** 0x000E8000 *****|**** RESERVED*********|
|**** 0x000E8000 ****|**** 16K *****|**** 0x000EC000 *****|***** user token******|
|**** 0x000EC000 ****|**** 32K *****|**** 0x000F4000 *****|***** tuya token******|
|**** 0x000F4000 ****|**** 8K ******|**** 0x000F6000 *****|***** oem cfg*********|
|**** 0x000F6000 ****|**** 32K *****|**** 0x000FE000 *****|***** sl toekn********|
|**** 0x000FE000 ****|**** 8K ******|**** 0x00100000 *****|***** EBL*************|
                          MG21 768K flash map
|**** start addr ****|**** size ****|***** end addr ******|***** description*****|
|**** 0x00000000 ****|**** 32K *****|**** 0x00008000 *****|***** bootloader******|
|**** 0x00008000 ****|**** 344K ****|**** 0x0005E000 *****|***** app code********|
|**** 0x0005E000 ****|**** 16K *****|**** 0x00062000 *****|*** diff backup*******|
|**** 0x00062000 ****|**** 248K ****|**** 0x000A0000 *****|***** ota*************|// diff ota size = ota; full ota size = ota+diff mange;
|**** 0x000A0000 ****|**** 16K *****|**** 0x000A4000 *****|**** diff mange*******|
|**** 0x000A4000 ****|**** 16K *****|**** 0x000A8000 *****|**** RESERVED*********|
|**** 0x000A8000 ****|**** 16K *****|**** 0x000AC000 *****|***** user token******|
|**** 0x000AC000 ****|**** 32K *****|**** 0x000B4000 *****|***** tuya token******|
|**** 0x000B4000 ****|**** 8K ******|**** 0x000B6000 *****|***** oem cfg*********|
|**** 0x000B6000 ****|**** 32K *****|**** 0x000BE000 *****|***** sl toekn********|
|**** 0x000BE000 ****|**** 8K ******|**** 0x000C0000 *****|***** EBL*************|
*/
/******************************************************************************/
/**                  internal macro definition following                     **/
/******************************************************************************/
// optional functions set
#define CHECK_DATA 0
#define DIRTY_DATA_WRITE_SUPPORT 0

#define NGX_ALIGN(d, a) (((d) + (a - 1)) & (~(a - 1)))

#define TUYA_FLASH_SIZE             (1024 * 1024UL)
#define TUYA_FLASH_PAGE_SIZE        (8 * 1024UL)
#define FLASH_ADDR_ALIGNMENT_SIZE   (4)

// boot loader flash info
#define BTL_CODE_PAGE_NUM           (4)
#define BTL_CODE_FLASH_ADDR         (0x00000000UL)
// user data flash info
#define USER_DATA_FLASH_ADDR        (0x0FE00000UL) // 1K only for EFR32MG21
#define USER_DATA_FLASH_SIZE        (1024UL)

#if defined(EFR32MG21A020F1024IM32)
// app code flash info
#define APP_CODE_PAGE_NUM           (62)
#define APP_CODE_FLASH_ADDR         (0x00008000UL)

// ota flash info
#define OTA_SPACE_PAGE_NUM          (46)
#define OTA_SPACE_FLASH_ADDR        (0x00088000UL)

// tuya kv flash info
#define TUYA_KV_PAGE_NUM            (6)
#define TUYA_KV_FLASH_ADDR          (0x000E8000UL)

// oem config flash info
#define OEM_CONFIG_PAGE_NUM         (1)
#define OEM_CONFIG_FLASH_ADDR       (0x000F4000UL)
#elif defined(EFR32MG21A020F768IM32)
// app code flash info
#define APP_CODE_PAGE_NUM           (43)
#define APP_CODE_FLASH_ADDR         (0x00008000UL)

// ota flash info
#define OTA_SPACE_PAGE_NUM          (33)
#define OTA_SPACE_FLASH_ADDR        (0x00062000UL)

// tuya kv flash info
#define TUYA_KV_PAGE_NUM            (6)
#define TUYA_KV_FLASH_ADDR          (0x000A8000UL)

// oem config flash info
#define OEM_CONFIG_PAGE_NUM         (1)
#define OEM_CONFIG_FLASH_ADDR       (0x000B4000UL)
#endif
/******************************************************************************/
/**                  internal type definition following                      **/
/******************************************************************************/

/******************************************************************************/
/**                  internal function declaration following                 **/
/******************************************************************************/

/******************************************************************************/
/**                  global veriables definition following                   **/
/******************************************************************************/
UINT8_T g_flash_lock = 0;
/******************************************************************************/
/**                  functions definition following                          **/
/******************************************************************************/

#if CHECK_DATA
STATIC BOOL_T __flash_data_check_erased(UINT32_T addr, UINT32_T size)
{
    UINT32_T i;
    UINT32_T word_len;
    UINT32_T check_addr = addr;
    UINT32_T check_size = size;
    UINT32_T remain_len = 0;

    while (check_addr % 4) {
        if (*(UINT8_T*)check_addr != 0xff) {
            return FALSE;
        }
        check_addr++;
        if (check_size) {
            check_size--;
        }
    }

    word_len = check_size / sizeof(UINT32_T);

    for (i = 0; i < word_len; i += 4) {
        if (*((UINT32_T*)check_addr) != 0xFFFFFFFFUL) {
            return FALSE;
        }
        check_addr += 4;
    }

    remain_len = check_size % sizeof(UINT32_T);

    while (remain_len) {
        if (*(UINT8_T*)check_addr != 0xff) {
            return FALSE;
        }
        check_addr++;
        if (remain_len) {
            remain_len--;
        }
    }
    return TRUE;
}

STATIC BOOL_T __flash_data_check_same(CONST UINT32_T addr,
                                      CONST UCHAR_T* buff,
                                      UINT32_T size) 
{
    UINT32_T i;

    for (i = 0; i < size; i++) {
        if (*((UINT8_T*)(addr + i)) != buff[i]) {
            return FALSE;
        }
    }

    return TRUE;
}
#endif

OPERATE_RET __flash_dirty_data_write(UINT32_T addr,
                                     CONST UINT8_T* buff,
                                     UINT32_T size)
{
    OPERATE_RET status = OPRT_OK;
    UINT32_T block_addr_start =
        (addr / TUYA_FLASH_PAGE_SIZE) * TUYA_FLASH_PAGE_SIZE;
    UINT16_T offset = addr % TUYA_FLASH_PAGE_SIZE;
    UINT16_T block_len = 0;

    UINT8_T temp_buff[TUYA_FLASH_PAGE_SIZE] = {0};

    if (buff == NULL) {
        return OPRT_INVALID_PARM;
    }
    if (size <= 0 && size > TUYA_FLASH_SIZE) {
        return OPRT_OS_ADAPTER_FLASH_ADDR_INVALID;
    }

    while (size > 0) {
        tkl_flash_read(block_addr_start, temp_buff, TUYA_FLASH_PAGE_SIZE);
        tkl_flash_erase(block_addr_start, TUYA_FLASH_PAGE_SIZE);

        block_len = (offset + size) > TUYA_FLASH_PAGE_SIZE
                        ? (TUYA_FLASH_PAGE_SIZE - offset)
                        : size;
        for (UINT16_T i = 0; i < block_len; i++) {
            temp_buff[offset + i] = *(buff + i);
        }

        size = size - block_len;
        buff = buff + block_len;

        status = (OPERATE_RET)MSC_WriteWord((UINT32_T*)block_addr_start, 
                                            (VOID_T*)temp_buff, 
                                            TUYA_FLASH_PAGE_SIZE);
        if (status != OPRT_OK) {
            return status;
        }

        block_addr_start = block_addr_start + TUYA_FLASH_PAGE_SIZE;
        offset = 0;
    }

    return status;
}

INT_T tkl_hal_flash_init(VOID_T)
{
    if (g_flash_lock == 0) {
        MSC_Init();
    }
    g_flash_lock++;

    return 0;
}

VOID_T tkl_hal_flash_deinit(VOID_T)
{
    if (g_flash_lock > 0) {
        g_flash_lock--;
        if (g_flash_lock == 0) {
            MSC_Deinit();
        }
    }
}

OPERATE_RET tkl_flash_read(UINT32_T addr, UCHAR_T *dst, UINT32_T size)
{
    if (dst == NULL || size == 0) {
        return OPRT_INVALID_PARM;
    }

    if (size > TUYA_FLASH_SIZE) {
        return OPRT_OS_ADAPTER_FLASH_ADDR_INVALID;
    }

    tkl_system_memcpy((UINT8_T*)dst, (UINT8_T*)addr, size);

    return OPRT_OK;
}

OPERATE_RET tkl_flash_write(UINT32_T addr, CONST UCHAR_T *src, UINT32_T size)
{
    UINT32_T length = size;
    UINT32_T swap_word = 0;
    OPERATE_RET status = OPRT_OK;
    UINT32_T v_addr = addr;
    UCHAR_T CONST* v_src = src;

    if (src == NULL || size == 0) {
        return OPRT_INVALID_PARM;
    }
    if (size == 0 && size > TUYA_FLASH_SIZE) {
        return OPRT_OS_ADAPTER_FLASH_ADDR_INVALID;
    }

    if (v_addr >= USER_DATA_FLASH_ADDR &&
        v_addr <= (USER_DATA_FLASH_ADDR + USER_DATA_FLASH_SIZE)) {
        // user data flash option
        if (size > USER_DATA_FLASH_SIZE) {
            return OPRT_INVALID_PARM;
        }
        SE_Response_t se_status = SE_writeUserData(0x00000000, (VOID_T *)src, size);
        if (se_status != SE_RESPONSE_OK) {
            status = OPRT_OS_ADAPTER_FLASH_WRITE_FAILED;
        }
    } else {
// common flash option
// dirty data write
#if CHECK_DATA
        if (FALSE == __flash_data_check_erased(v_addr, size)) {
            status = OPRT_OS_ADAPTER_NOT_ERASE;
#if DIRTY_DATA_WRITE_SUPPORT
            status = __flash_dirty_data_write(v_addr, src, size);
#endif
            return status;
        }
#endif

        UINT8_T diff_num = NGX_ALIGN(v_addr, FLASH_ADDR_ALIGNMENT_SIZE) - v_addr;
        // hal_flash_debug("diff_num %d\r\n",diff_num);
        // handle addr not aligned with FLASH_ADDR_ALIGNMENT_SIZE bytes
        if (diff_num) {
            UINT8_T swap_num;
            if (diff_num < length) {
                swap_num = diff_num;
            } else {
                swap_num = length;
            }

            v_addr -= (FLASH_ADDR_ALIGNMENT_SIZE - diff_num);
            tkl_flash_read(v_addr, (UINT8_T*)&(swap_word),
                           FLASH_ADDR_ALIGNMENT_SIZE);
            tkl_system_memcpy((((UINT8_T*)&swap_word) + (FLASH_ADDR_ALIGNMENT_SIZE - diff_num)), src, swap_num);
            status = (OPERATE_RET)MSC_WriteWord((UINT32_T*)addr, 
                                                (VOID_T*)&swap_word, 
                                                FLASH_ADDR_ALIGNMENT_SIZE);
            if (status != OPRT_OK) {
                return status;
            }

            v_addr += FLASH_ADDR_ALIGNMENT_SIZE;
            length -= swap_num;
            src += swap_num;
        }

        UINT32_T word_len = length / FLASH_ADDR_ALIGNMENT_SIZE;
        UINT32_T byte_len = length % FLASH_ADDR_ALIGNMENT_SIZE;
        // hal_flash_debug("word len %d, byte len %d, status %d\r\n",word_len,
        // byte_len, status); data byte size more than FLASH_ADDR_ALIGNMENT_SIZE
        // bytes
        if (word_len > 0) {
            status = (OPERATE_RET)MSC_WriteWord((UINT32_T*)v_addr, 
                                                (VOID_T*)src,
                                                word_len * FLASH_ADDR_ALIGNMENT_SIZE);
            if (status != OPRT_OK) {
                return status;
            }

            v_addr = (v_addr + (word_len * FLASH_ADDR_ALIGNMENT_SIZE));
            src += (word_len * FLASH_ADDR_ALIGNMENT_SIZE);
        }

        // handle size not aligned with FLASH_ADDR_ALIGNMENT_SIZE bytes
        if ((byte_len > 0) && (status == OPRT_OK)) {
            swap_word = 0xFFFFFFFF;
            tkl_system_memcpy(&swap_word, src, byte_len);

            status = (OPERATE_RET)MSC_WriteWord((UINT32_T*)v_addr, 
                                                (VOID_T*)&swap_word, 
                                                FLASH_ADDR_ALIGNMENT_SIZE);
            if (status != OPRT_OK) {
                return status;
            }
        }

// check data write correct
#if CHECK_DATA
        if (status == OPRT_OK) {
            if (__flash_data_check_same(addr, v_src, size) == FALSE) {
                status = OPRT_OS_ADAPTER_DATA_CHACK_FAIL;
            }
        }
#endif
    }

    return status;
}

OPERATE_RET tkl_flash_erase(CONST UINT32_T addr, CONST UINT32_T size)
{
    UINT8_T i;
    UINT32_T new_addr;
    OPERATE_RET status = OPRT_OK;

    if (size == 0) {
        return OPRT_OS_ADAPTER_FLASH_ADDR_INVALID;
    }

    UINT32_T v_real_size = 0;
    UINT32_T v_real_start_addr = addr - (addr % TUYA_FLASH_PAGE_SIZE);

    //erase length is not Tuya_ FLASH_ PAGE_ Multiple of size
    if (size % TUYA_FLASH_PAGE_SIZE != 0) {
        v_real_size = (size / TUYA_FLASH_PAGE_SIZE + 1) * TUYA_FLASH_PAGE_SIZE;
    } else {
        v_real_size = size;
    }

    //processing of erasing non page alignment of start address
    if ((addr + size) > (v_real_start_addr + v_real_size)) {
        v_real_size += TUYA_FLASH_PAGE_SIZE;
    }

    if (addr >= USER_DATA_FLASH_ADDR &&
        addr <= (USER_DATA_FLASH_ADDR + USER_DATA_FLASH_SIZE)) {
        // user data flash option
        SE_Response_t se_status = SE_eraseUserData();  // only for MG21 user data flash
        if (se_status != SE_RESPONSE_OK) {
            status = OPRT_OS_ADAPTER_FLASH_WRITE_FAILED;
        }
    } else {
        // common flash option
        new_addr = addr - (addr % TUYA_FLASH_PAGE_SIZE);

        for (i = 0; i < v_real_size / TUYA_FLASH_PAGE_SIZE; i++) {
            status = (OPERATE_RET)MSC_ErasePage((UINT32_T*)new_addr);
            if (status != OPRT_OK) {
                break;
            }
#if CHECK_DATA
            if (status == OPRT_OK) {
                if (FALSE ==
                    __flash_data_check_erased(new_addr, TUYA_FLASH_PAGE_SIZE)) {
                    status = OPRT_OS_ADAPTER_NOT_ERASE;
                }
            }
#endif
            new_addr += TUYA_FLASH_PAGE_SIZE;
        }
    }

    return status;
}

OPERATE_RET tkl_flash_lock(CONST UINT32_T addr, CONST UINT32_T size)
{
    return OPRT_INVALID_PARM;
}


OPERATE_RET tkl_flash_unlock(CONST UINT32_T addr, CONST UINT32_T size)
{
    return OPRT_INVALID_PARM;
}

OPERATE_RET tkl_flash_get_one_type_info(TUYA_FLASH_TYPE_E type, TUYA_FLASH_BASE_INFO_T* info)
{
    if (NULL == info) {
        return OPRT_INVALID_PARM;
    }
/*
                          MG21 1M flash map new OTA 
|**** 0x00000000 ****|**** 32K *****|**** 0x00008000 *****|***** bootloader******|
|**** 0x00008000 ****|**** 496K ****|**** 0x00084000 *****|***** app code********|
|**** 0x00084000 ****|**** 16K *****|**** 0x00088000 *****|*** diff backup*******|
|**** 0x00088000 ****|**** 352K ****|**** 0x000E0000 *****|***** ota*************|
|**** 0x000E0000 ****|**** 16K *****|**** 0x000E4000 *****|**** diff mange*******|
|**** 0x000E4000 ****|**** 16K *****|**** 0x000E8000 *****|**** RESERVED*********|
|**** 0x000E8000 ****|**** 16K *****|**** 0x000EC000 *****|***** user token******|
|**** 0x000EC000 ****|**** 32K *****|**** 0x000F4000 *****|***** tuya token******|
|**** 0x000F4000 ****|**** 8K ******|**** 0x000F6000 *****|***** oem cfg*********|
|**** 0x000F6000 ****|**** 32K *****|**** 0x000FE000 *****|***** sl toekn********|
|**** 0x000FE000 ****|**** 8K ******|**** 0x00100000 *****|***** EBL*************|

                          MG21 768K flash map
|**** start addr ****|**** size ****|***** end addr ******|***** description*****|
|**** 0x00000000 ****|**** 32K *****|**** 0x00008000 *****|***** bootloader******|
|**** 0x00008000 ****|**** 344K ****|**** 0x0005E000 *****|***** app code********|
|**** 0x0005E000 ****|**** 16K *****|**** 0x00062000 *****|*** diff backup*******|
|**** 0x00062000 ****|**** 248K ****|**** 0x000A0000 *****|***** ota*************|
|**** 0x000A0000 ****|**** 16K *****|**** 0x000A4000 *****|**** diff mange*******|
|**** 0x000A4000 ****|**** 16K *****|**** 0x000A8000 *****|**** RESERVED*********|
|**** 0x000A8000 ****|**** 16K *****|**** 0x000AC000 *****|***** user token******|
|**** 0x000AC000 ****|**** 32K *****|**** 0x000B4000 *****|***** tuya token******|
|**** 0x000B4000 ****|**** 8K ******|**** 0x000B6000 *****|***** oem cfg*********|
|**** 0x000B6000 ****|**** 32K *****|**** 0x000BE000 *****|***** sl toekn********|
|**** 0x000BE000 ****|**** 8K ******|**** 0x000C0000 *****|***** EBL*************|
*/

    info->partition_num = 1;
    info->partition[0].block_size = TUYA_FLASH_PAGE_SIZE;
    switch (type) {
        case TUYA_FLASH_TYPE_BTL0: {
            info->partition[0].start_addr = BTL_CODE_FLASH_ADDR;
            info->partition[0].size = BTL_CODE_PAGE_NUM*TUYA_FLASH_PAGE_SIZE;
        } break;
        case TUYA_FLASH_TYPE_BTL1: {
        } break;
        case TUYA_FLASH_TYPE_APP: {
            info->partition[0].start_addr = APP_CODE_FLASH_ADDR;
            info->partition[0].size = APP_CODE_PAGE_NUM*TUYA_FLASH_PAGE_SIZE;
        } break;
        case TUYA_FLASH_TYPE_OTA: {
            info->partition[0].start_addr = OTA_SPACE_FLASH_ADDR;
            info->partition[0].size = OTA_SPACE_PAGE_NUM*TUYA_FLASH_PAGE_SIZE;
        } break;
        case TUYA_FLASH_TYPE_KV_DATA: {
            info->partition[0].start_addr = TUYA_KV_FLASH_ADDR;
            info->partition[0].size = TUYA_KV_PAGE_NUM*TUYA_FLASH_PAGE_SIZE;
        } break;
        case TUYA_FLASH_TYPE_USER0: {
            info->partition[0].start_addr = OEM_CONFIG_FLASH_ADDR;
            info->partition[0].size = OEM_CONFIG_PAGE_NUM*TUYA_FLASH_PAGE_SIZE;
        } break;
        case TUYA_FLASH_TYPE_USER1: {
            info->partition[0].start_addr = USER_DATA_FLASH_ADDR; // efr32-user-flash
            info->partition[0].size = USER_DATA_FLASH_SIZE;
            info->partition[0].block_size = USER_DATA_FLASH_SIZE;
        } break;
        default: {
            return OPRT_NOT_SUPPORTED;
        }
    }
    return OPRT_OK;
}
/******************************************************************************/
/**                              EOF                                         **/
/******************************************************************************/
