/**
 * @file tkl_pwm.c
 * @brief This is tuya tkl pwm src file
 * @version 1.0
 * @date 2021-09-10
 *
 * @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
 *
 */
#include "em_cmu.h"
#include "em_timer.h"

#include "tkl_memory.h"
#include "tkl_pwm.h"
/******************************************************************************/
/**                  module description following                            **/
/******************************************************************************/
/* Main Features below:
polarity=TUYA_PWM_POSITIVE && duty=0,output low level

TUYA_PWM_POSITIVE: duty corresponds to high level
TUYA_PWM_NEGATIVE: duty corresponds to low level
*/
/******************************************************************************/
/**                  internal macro definition following                     **/
/******************************************************************************/
// when debug,disable it
#define tkl_pwm_release_en 0

#define INVALID_INDEX           0xFFFF
// pwm timer id
#define PWM_TIMER0_ID           0
#define PWM_TIMER1_ID           1

#define PWM_DEFAULT_PRECISION (1000)

#define PWM_TIMER0_ACTIVE_MASK (0x07u)
#define PWM_TIMER1_ACTIVE_MASK (0x38u)

// pwm default gpio id
#define PWM_MAX_CHANNEL         6
#define PWM1_CH0_GPIO_ID        7  //PB0
#define PWM1_CH1_GPIO_ID        3  //PA3
#define PWM1_CH2_GPIO_ID        0  //PA0
#define PWM1_CH3_GPIO_ID        8  //PB1
#define PWM1_CH4_GPIO_ID        4  //PA4
#define PWM1_CH5_GPIO_ID        9  //PB2

#define __PWM_BIT_CHECK(state, ch)  ((state) & (1 << ch))
#define __PWM_BIT_SET(state, ch)    (state |= (1 << ch))
#define __PWM_BIT_CLR(state, ch)    (state &= ~(1 << ch))

/******************************************************************************/
/**                  internal type definition following                      **/
/******************************************************************************/

/******************************************************************************/
/**                  internal function declaration following                 **/
/******************************************************************************/

/******************************************************************************/
/**                  global veriables definition following                   **/
/******************************************************************************/

typedef struct {
  TUYA_PWM_BASE_CFG_T cfg;
  UINT_T precision;  // (duty max)
  UINT8_T idle_level;
  UINT32_T tick_per_period;
} __TKL_PWM_PRAVATE_CFG_T;

STATIC struct {
  __TKL_PWM_PRAVATE_CFG_T pwm_info[PWM_MAX_CHANNEL];
  union {
    UCHAR_T all;
    struct {
      UCHAR_T is_timer0_ative : 1;
      UCHAR_T is_timer1_ative : 1;
      UCHAR_T reserved : 6;
    } area;
  } timer_state;
  
  union {
    UCHAR_T all;
    struct {
      UCHAR_T is_ch0_init : 1;
      UCHAR_T is_ch1_init : 1;
      UCHAR_T is_ch2_init : 1;
      UCHAR_T is_ch3_init : 1;
      UCHAR_T is_ch4_init : 1;
      UCHAR_T is_ch5_init : 1;
      UCHAR_T reserved : 2;
      
    } area;
  } pwm_ch_state;
  union {
    UCHAR_T all;
    struct {
      UCHAR_T is_ch0_stop : 1;
      UCHAR_T is_ch1_stop : 1;
      UCHAR_T is_ch2_stop : 1;
      UCHAR_T is_ch3_stop : 1;
      UCHAR_T is_ch4_stop : 1;
      UCHAR_T is_ch5_stop : 1;
      UCHAR_T reserved : 2;
      
    } area;
  } pwm_ch_stop;
  UCHAR_T pwm_ch_num;
  
} sg_pwm_cfg_tbl = {
  .pwm_ch_num = 0,
  .timer_state.all = 0,
  .pwm_ch_state.all = 0,
  .pwm_ch_stop.all=0,
};

// default pwm_port-gpio map
STATIC struct {
  UINT32_T gpio_id;
} sg_pwm_map[PWM_MAX_CHANNEL] = {
  // use timer0
  {PWM1_CH0_GPIO_ID},
  {PWM1_CH1_GPIO_ID},
  {PWM1_CH2_GPIO_ID},
  // use timer1
  {PWM1_CH3_GPIO_ID},
  {PWM1_CH4_GPIO_ID},
  {PWM1_CH5_GPIO_ID},
};

const TIMER_Prescale_TypeDef g_pwm_div_freq_list[] = {
    timerPrescale1,
    timerPrescale2,
    timerPrescale4,
    timerPrescale8,
    timerPrescale16,
    timerPrescale32,
    timerPrescale64,
    timerPrescale128,
    timerPrescale256,
    timerPrescale512,
    timerPrescale1024,
};

STATIC UINT32_T sg_ticksPerPeriod=0;
STATIC UINT32_T sg_peripheral_clock=0;

extern UINT8_T __get_gpio_pin_num(UINT32_T pin_id);
extern UINT8_T __get_gpio_port_num(UINT32_T pin_id);

/******************************************************************************/
/**                  functions definition following                          **/
/******************************************************************************/
STATIC UINT32_T __get_pwm_gpio_from_port(UINT32_T pwm_id)
{
  return sg_pwm_map[pwm_id].gpio_id;
}

STATIC UINT8_T __get_pwm_timer_sn(UINT32_T pwm_id)
{
  if(pwm_id < 3) {
    return PWM_TIMER0_ID;
  }
  else {
    return PWM_TIMER1_ID;
  }
}

STATIC UINT8_T __get_pwm_cct_sn(UINT32_T pwm_id)
{
  return pwm_id % 3;
}

STATIC bool_t __check_timer0_active(VOID_T)
{
  return sg_pwm_cfg_tbl.timer_state.area.is_timer0_ative;
}
STATIC bool_t __check_timer1_active(VOID_T)
{
  return sg_pwm_cfg_tbl.timer_state.area.is_timer1_ative;
}
static UINT8_T __peripheral_clock_self_adaption(UINT16_T pwm_freq)
{
    UINT32_T v_current_ticksPerPeriod=0;
    UINT8_T v_freq_index = 0;

    sg_peripheral_clock = CMU_ClockFreqGet(cmuClock_EM01GRPACLK);

    while(1) {
        v_current_ticksPerPeriod = sg_peripheral_clock/pwm_freq;
        if(v_current_ticksPerPeriod < 65534UL) {
            break;
        }
        sg_peripheral_clock >>=1;
        v_freq_index++;
    }

    if(v_freq_index > 10) {
        return 0xFF; //failed.
    }

    sg_ticksPerPeriod = v_current_ticksPerPeriod;

    return v_freq_index;
}
#if tkl_pwm_release_en
STATIC VOID __pwm_gpio_disable_tdio(VOID) {
  BUS_RegBitWrite(&(GPIO->DBGROUTEPEN), _GPIO_DBGROUTEPEN_TDIPEN_SHIFT,
                  0); /*disable TDI PA4*/
  BUS_RegBitWrite(&(GPIO->DBGROUTEPEN), _GPIO_DBGROUTEPEN_TDOPEN_SHIFT,
                  0); /*disable TDO PA3*/
  BUS_RegBitWrite(&(GPIO->TRACEROUTEPEN),
                  _GPIO_TRACEROUTEPEN_TRACECLKPEN_SHIFT,
                  0); /*disable TRACCLK PA4*/
  BUS_RegBitWrite(&(GPIO->TRACEROUTEPEN),
                  _GPIO_TRACEROUTEPEN_TRACEDATA0PEN_SHIFT,
                  0); /*disable TRACEDATA0 PA3*/
  BUS_RegBitWrite(&(GPIO->TRACEROUTEPEN), _GPIO_TRACEROUTEPEN_SWVPEN_SHIFT,
                  0); /*disable SWV PA3*/
}

STATIC VOID __pwm_gpio_disable_debug(GPIO_Port_TypeDef port, INT_T pin) {
  if (port == gpioPortA && pin == 1 || port == gpioPortA && pin == 2) {
    BUS_RegBitWrite(&(GPIO->DBGROUTEPEN),
                    _GPIO_DBGROUTEPEN_SWCLKTCKPEN_SHIFT,
                    0); /*disable SWCLK PA1*/
    BUS_RegBitWrite(&(GPIO->DBGROUTEPEN),
                    _GPIO_DBGROUTEPEN_SWDIOTMSPEN_SHIFT,
                    0); /*disable SWD PA2*/
  }
}
STATIC VOID __pwm_gpio_clear_all_special_func(GPIO_Port_TypeDef port,
                                              INT_T pin) {
                                                __pwm_gpio_disable_tdio();
                                                __pwm_gpio_disable_debug(port, pin);
                                              }
#endif


// #define _GPIO_TIMER_CC0ROUTE_PORT_SHIFT   _GPIO_TIMER_CC0ROUTE_PORT_SHIFT // all 0
// #define _GPIO_TIMER_CC0ROUTE_PIN_SHIFT    _GPIO_TIMER_CC0ROUTE_PIN_SHIFT  // all 16
STATIC VOID_T __gpio_timer_ccrout_mapping_enable(UINT8_T timer_id, UINT8_T cct_id, UINT32_T gpio_id)
{
  UINT8_T v_port, v_pin;
  v_pin = __get_gpio_pin_num(gpio_id);
  v_port = __get_gpio_port_num(gpio_id);
  
  if(cct_id==0){        
    GPIO->TIMERROUTE[timer_id].CC0ROUTE |= ((UINT32_T)(v_port) << _GPIO_TIMER_CC0ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC0ROUTE_PIN_SHIFT);
    GPIO->TIMERROUTE[timer_id].ROUTEEN |= (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
  }else if(cct_id==1){
    GPIO->TIMERROUTE[timer_id].CC1ROUTE |= ((UINT32_T)(v_port) << _GPIO_TIMER_CC1ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC1ROUTE_PIN_SHIFT);
    GPIO->TIMERROUTE[timer_id].ROUTEEN |= (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
  }else if(cct_id==2){
    GPIO->TIMERROUTE[timer_id].CC2ROUTE |= ((UINT32_T)(v_port) << _GPIO_TIMER_CC2ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC2ROUTE_PIN_SHIFT);
    GPIO->TIMERROUTE[timer_id].ROUTEEN |= (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
  }
  
}

STATIC VOID_T __timer_ccrout_mapping_to_gpio_disable(UINT8_T timer_id, UINT8_T cct_id, UINT32_T gpio_id)
{
  UINT8_T v_port, v_pin;
  v_pin = __get_gpio_pin_num(gpio_id);
  v_port = __get_gpio_port_num(gpio_id);
  
  if(cct_id==0){        
    GPIO->TIMERROUTE[timer_id].ROUTEEN &= ~ (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
    GPIO->TIMERROUTE[timer_id].CC0ROUTE &= ~ ((UINT32_T)(v_port) << _GPIO_TIMER_CC0ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC0ROUTE_PIN_SHIFT);
  }else if(cct_id==1){
    GPIO->TIMERROUTE[timer_id].ROUTEEN &= ~ (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
    GPIO->TIMERROUTE[timer_id].CC1ROUTE &= ~ ((UINT32_T)(v_port) << _GPIO_TIMER_CC1ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC1ROUTE_PIN_SHIFT);
  }else if(cct_id==2){
    GPIO->TIMERROUTE[timer_id].ROUTEEN &= ~ (0x1UL << cct_id);  //GPIO_TIMER_ROUTEEN_CC0PEN;
    GPIO->TIMERROUTE[timer_id].CC2ROUTE &= ~ ((UINT32_T)(v_port) << _GPIO_TIMER_CC2ROUTE_PORT_SHIFT) | \
      ((UINT32_T)(v_pin) << _GPIO_TIMER_CC2ROUTE_PIN_SHIFT);
  }
} 

OPERATE_RET tkl_pwm_mapping_to_gpio(UINT32_T ch_id, UINT32_T gpio_id)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  sg_pwm_map[ch_id].gpio_id = gpio_id;
  
  return OPRT_OK;
}


OPERATE_RET tkl_pwm_init(TUYA_PWM_NUM_E ch_id, CONST TUYA_PWM_BASE_CFG_T *cfg)
{
  // UINT32_T v_ticksPerPeriod32=0;
  if (ch_id >= PWM_MAX_CHANNEL || NULL == cfg || cfg->frequency==0) {
    return OPRT_INVALID_PARM;
  }
  if (__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  
  // enable pwm clock and gpio clock
  if (sg_pwm_cfg_tbl.timer_state.all == 0) {
    CMU_ClockEnable(cmuClock_EM01GRPACLK, true);
    CMU_ClockEnable(cmuClock_GPIO, true);
  }

#if tkl_pwm_release_en
  __pwm_gpio_clear_all_special_func((GPIO_Port_TypeDef)v_port, v_pin);
#endif
  
  // enable clock for timer module
  if (v_timer_sn == PWM_TIMER0_ID && !__check_timer0_active()) {
    CMU_ClockEnable(cmuClock_TIMER0, true);
  } 
  else if (v_timer_sn == PWM_TIMER1_ID && !__check_timer1_active()) {
    CMU_ClockEnable(cmuClock_TIMER1, true);
  }

  if(sg_pwm_cfg_tbl.pwm_info[ch_id].precision==0){
    sg_pwm_cfg_tbl.pwm_info[ch_id].precision=PWM_DEFAULT_PRECISION;
  }

  UINT8_T v_div_index=__peripheral_clock_self_adaption(cfg->frequency);
  if(v_div_index == 0xFF) {
        return 0;
  }

  //check freq and pricision
  if(sg_pwm_cfg_tbl.pwm_info[ch_id].precision>sg_ticksPerPeriod){
    return OPRT_EXCEED_UPPER_LIMIT;
  }
 
  sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period = sg_ticksPerPeriod;
  tkl_system_memcpy(&sg_pwm_cfg_tbl.pwm_info[ch_id].cfg, cfg, sizeof(TUYA_PWM_BASE_CFG_T));
  
  TIMER_InitCC_TypeDef timerCCInit = {
    .eventCtrl = timerEventEveryEdge,
    .edge = timerEdgeBoth,
    .prsSel = timerPRSSELCh0,
    .cufoa = timerOutputActionNone,
    .cofoa = timerOutputActionNone,
    .cmoa = timerOutputActionToggle,
    .mode = timerCCModePWM,
    .filter = false,
    .prsInput = false,
    //coist: true==>counter stop high,false counter stop low
    .coist = (sg_pwm_cfg_tbl.pwm_info[ch_id].idle_level) ? true: false,
    // Positive or negative, invert = 0:Positive, 1:negative
    .outInvert = !(cfg->polarity),
  };
  
  // update the Settings PWM parameters effective
  if (v_timer_sn == PWM_TIMER0_ID && v_cct_sn < 3) {
    TIMER_InitCC(TIMER0, v_cct_sn, &timerCCInit);
    TIMER_CompareBufSet(TIMER0, v_cct_sn, cfg->duty * (sg_ticksPerPeriod) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
     TIMER_CompareSet(TIMER0, v_cct_sn, cfg->duty * (sg_ticksPerPeriod) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
  }
  else if (v_timer_sn == PWM_TIMER1_ID && v_cct_sn < 3) {
    TIMER_InitCC(TIMER1, v_cct_sn, &timerCCInit);
    TIMER_CompareBufSet(TIMER1, v_cct_sn, cfg->duty * (sg_ticksPerPeriod) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
     TIMER_CompareSet(TIMER1, v_cct_sn, cfg->duty * (sg_ticksPerPeriod) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
  }
  
  TIMER_Init_TypeDef timerInit = {
    .enable = true,
    .debugRun = true,
    .prescale = g_pwm_div_freq_list[v_div_index],
    .clkSel = timerClkSelHFPerClk,
    .fallAction = timerInputActionNone,
    .riseAction = timerInputActionNone,
    .mode = timerModeUp,
    .dmaClrAct = false,
    .quadModeX4 = false,
    .oneShot = false,
    .sync = false,
  };


    //note:
    // when multi pwm channel drived by a same timer,but the frequency is
    // different,latest parameter is effectived
  if (v_timer_sn == PWM_TIMER0_ID) {
    TIMER_TopSet(TIMER0, sg_ticksPerPeriod-1);
     TIMER_TopBufSet(TIMER0, sg_ticksPerPeriod-1);

    if (!__check_timer0_active()) {
      TIMER_Init(TIMER0, &timerInit);
      sg_pwm_cfg_tbl.timer_state.area.is_timer0_ative = 1;
    }
    TIMER_Enable(TIMER0, TRUE);
  }
  else if (v_timer_sn == PWM_TIMER1_ID) {
    TIMER_TopSet(TIMER1, sg_ticksPerPeriod-1);
     TIMER_TopBufSet(TIMER1, sg_ticksPerPeriod-1);
    if (!__check_timer1_active()) {
      TIMER_Init(TIMER1, &timerInit);
      sg_pwm_cfg_tbl.timer_state.area.is_timer1_ative = 1;
    }
    TIMER_Enable(TIMER1, TRUE);
  }
  
  __PWM_BIT_SET(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id);
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_deinit(TUYA_PWM_NUM_E ch_id)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT32_T gpio_id = __get_pwm_gpio_from_port((UINT32_T)ch_id);
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  
  TIMER_InitCC_TypeDef timerCCInit = {
    .eventCtrl = timerEventEveryEdge,
    .edge = timerEdgeBoth,
    .prsSel = timerPRSSELCh0,
    .cufoa = timerOutputActionNone,
    .cofoa = timerOutputActionNone,
    .cmoa = timerOutputActionToggle,
    .mode = timerCCModeOff,
    .filter = false,
    .prsInput = false,
    .coist = false,
    .outInvert = 0,  // Positive or negative, invert = 0:Positive, 1:negative
  };
  __PWM_BIT_CLR(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id);
  if (v_timer_sn == PWM_TIMER0_ID) {
    TIMER_CompareBufSet(TIMER0, v_cct_sn, 0);
    
    TIMER_InitCC(TIMER0, v_cct_sn, &timerCCInit);
    if (sg_pwm_cfg_tbl.pwm_ch_state.all & PWM_TIMER0_ACTIVE_MASK) {
      TIMER_Enable(TIMER0, TRUE);
    } else {
      TIMER_Enable(TIMER0, FALSE);
      TIMER_Reset(TIMER0);
      sg_pwm_cfg_tbl.timer_state.area.is_timer0_ative = 0;
    }
  }
  else if (v_timer_sn == PWM_TIMER1_ID) {
    TIMER_CompareBufSet(TIMER1, v_cct_sn, 0);
    
    TIMER_InitCC(TIMER1, v_cct_sn, &timerCCInit);
    if (sg_pwm_cfg_tbl.pwm_ch_state.all & PWM_TIMER1_ACTIVE_MASK) {
      TIMER_Enable(TIMER1, TRUE);
    } else {
      TIMER_Enable(TIMER1, FALSE);
      TIMER_Reset(TIMER1);
      sg_pwm_cfg_tbl.timer_state.area.is_timer1_ative = 0;
    }
  }
  
  // set gpio remap register
  __timer_ccrout_mapping_to_gpio_disable(v_timer_sn, v_cct_sn, gpio_id);
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_start(TUYA_PWM_NUM_E ch_id)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  //wait for ccn validate:1ms
 UINT32_T v_clock=CMU_ClockFreqGet(cmuClock_EM01GRPACLK);
 for(UINT32_T i=0;i<(v_clock/1000);i++);

  UINT8_T v_port, v_pin;
  for (UINT8_T i = 0; i <PWM_MAX_CHANNEL;i++)
  {
    if(sg_pwm_cfg_tbl.pwm_ch_state.all&(1<<i))
    {
      UINT32_T gpio_id = __get_pwm_gpio_from_port(i);
      v_pin = __get_gpio_pin_num(gpio_id);
      v_port = __get_gpio_port_num(gpio_id);
      
      UINT8_T v_timer_sn = __get_pwm_timer_sn(i);
      UINT8_T v_cct_sn = __get_pwm_cct_sn(i);
      
      //wait for ccn validate:1ms
      for(UINT32_T i=0;i<10000;i++);
      
      // set gpio remap register
      GPIO_PinModeSet((GPIO_Port_TypeDef)(v_port), (UINT32_T)(v_pin), gpioModePushPull, sg_pwm_cfg_tbl.pwm_info[i].idle_level);
      __gpio_timer_ccrout_mapping_enable(v_timer_sn, v_cct_sn, gpio_id);
    }
  }
  
  if (__check_timer0_active()) {
    TIMER_Enable(TIMER0, TRUE);
    // Reset counting
    TIMER_CounterSet(TIMER0, 0);
  }
  
  if (__check_timer1_active()) {
    TIMER_Enable(TIMER1, TRUE);
    // Reset counting
    TIMER_CounterSet(TIMER1, 0);
  }
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_stop(TUYA_PWM_NUM_E ch_id)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  if (__check_timer0_active()) {
    TIMER_Enable(TIMER0, FALSE);
    // Reset counting
    TIMER_CounterSet(TIMER0, 0);
  }
  
  if (__check_timer1_active()) {
    TIMER_Enable(TIMER1, FALSE);
    // Reset counting
    TIMER_CounterSet(TIMER1, 0);
  }
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_duty_set(TUYA_PWM_NUM_E ch_id, UINT32_T duty)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  
  UINT32_T precision = sg_pwm_cfg_tbl.pwm_info[ch_id].precision;
  if(duty>precision){
    return OPRT_INVALID_PARM;
  }
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.duty != duty) {
    // update the Settings PWM parameters effective
    if (v_timer_sn == PWM_TIMER0_ID) {
      TIMER_CompareBufSet(TIMER0, v_cct_sn,
                          duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
    }
    
    if (v_timer_sn == PWM_TIMER1_ID) {
      TIMER_CompareBufSet(TIMER1, v_cct_sn,
                          duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
    }
    sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.duty = duty;
  }
  
  return OPRT_OK;
}
/**
* @brief pwm frequency set
* 
* @param[in] ch_id: pwm channal id, id index starts at 0
* @param[in] precision: pwm precision
*
* @return OPRT_OK on success. Others on error, please refer to tuya_error_code.h
*/
OPERATE_RET tkl_pwm_precision_set(TUYA_PWM_NUM_E ch_id, UINT32_T precision)
{
  if (ch_id >= PWM_MAX_CHANNEL || precision ==0) {
    return OPRT_INVALID_PARM;
  }
  if (__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  sg_pwm_cfg_tbl.pwm_info[ch_id].precision = precision;

  return OPRT_OK;
}

OPERATE_RET tkl_pwm_frequency_set(TUYA_PWM_NUM_E ch_id, UINT32_T frequency)
{
  if (ch_id >= PWM_MAX_CHANNEL ||  frequency==0) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  
  UINT32_T precision = sg_pwm_cfg_tbl.pwm_info[ch_id].precision;

  UINT8_T v_div_index=__peripheral_clock_self_adaption(frequency);
  if(v_div_index == 0xFF) {
        return 0;
  }
  
  if(precision>sg_ticksPerPeriod){
    return OPRT_EXCEED_UPPER_LIMIT;
  }
  
  UINT32_T v_duty=sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.duty;
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.frequency != frequency) {
    // update information Configure CC channel 0 ~ sum
    sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period = sg_ticksPerPeriod;
    if (v_timer_sn == PWM_TIMER0_ID) {
      TIMER_TopSet(TIMER0, sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period-1);
      // TIMER_TopBufSet(TIMER0, sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period);
      TIMER_CompareBufSet(TIMER0, v_cct_sn,
                          v_duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
      // TIMER_CompareSet(TIMER0, v_cct_sn,
      //                  v_duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
    }
    else if (v_timer_sn == PWM_TIMER1_ID) {
      TIMER_TopSet(TIMER1, sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period-1);
      // TIMER_TopBufSet(TIMER1, sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period);
      TIMER_CompareBufSet(TIMER1, v_cct_sn,
                          v_duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
      // TIMER_CompareSet(TIMER1, v_cct_sn,
      //                  v_duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / precision);
    }
    sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.frequency = frequency;
  }
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_polarity_set(TUYA_PWM_NUM_E ch_id, TUYA_PWM_POLARITY_E polarity)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.polarity != polarity) {
    TIMER_InitCC_TypeDef timerCCInit = {
      .eventCtrl = timerEventEveryEdge,
      .edge = timerEdgeBoth,
      .prsSel = timerPRSSELCh0,
      .cufoa = timerOutputActionNone,
      .cofoa = timerOutputActionNone,
      .cmoa = timerOutputActionToggle,
      .mode = timerCCModePWM,
      .filter = false,
      .prsInput = false,
      //coist: true==>counter stop high,false counter stop low
      .coist = (sg_pwm_cfg_tbl.pwm_info[ch_id].idle_level) ? false:true,// : false,
      // Positive or negative, invert = 0:Positive, 1:negative
      .outInvert = !(polarity),
    };
    if (v_timer_sn == PWM_TIMER0_ID) {
      TIMER_InitCC(TIMER0, v_cct_sn, &timerCCInit);
      TIMER_Enable(TIMER0, TRUE);
    }
    else if (v_timer_sn == PWM_TIMER1_ID) {
      TIMER_InitCC(TIMER1, v_cct_sn, &timerCCInit);
      TIMER_Enable(TIMER1, TRUE);
    }
    sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.polarity = polarity;
  }
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_idle_level_set(TUYA_PWM_NUM_E ch_id, UINT8_T idle_level)
{
  if (ch_id >= PWM_MAX_CHANNEL) {
    return OPRT_INVALID_PARM;
  }
  
  sg_pwm_cfg_tbl.pwm_info[ch_id].idle_level = idle_level;
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_info_set(TUYA_PWM_NUM_E ch_id, CONST TUYA_PWM_BASE_CFG_T *info)
{
  if (ch_id >= PWM_MAX_CHANNEL || NULL == info || info->frequency==0) {
    return OPRT_INVALID_PARM;
  }
  
  if (info->duty > sg_pwm_cfg_tbl.pwm_info[ch_id].precision){
    return OPRT_INVALID_PARM;
  }
  
  UINT8_T v_timer_sn = __get_pwm_timer_sn((UINT32_T)ch_id);
  UINT8_T v_cct_sn = __get_pwm_cct_sn((UINT32_T)ch_id);
  if (!__PWM_BIT_CHECK(sg_pwm_cfg_tbl.pwm_ch_state.all, ch_id)) {
    return OPRT_COM_ERROR;
  }
  
  UINT8_T v_div_index=__peripheral_clock_self_adaption(info->frequency);
  if(v_div_index == 0xFF) {
        return 0;
  }

  if(sg_pwm_cfg_tbl.pwm_info[ch_id].precision>sg_ticksPerPeriod){
    return OPRT_EXCEED_UPPER_LIMIT;
  }
  
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.polarity != info->polarity) {
    TIMER_InitCC_TypeDef timerCCInit = {
      .eventCtrl = timerEventEveryEdge,
      .edge = timerEdgeBoth,
      .prsSel = timerPRSSELCh0,
      .cufoa = timerOutputActionNone,
      .cofoa = timerOutputActionNone,
      .cmoa = timerOutputActionToggle,
      .mode = timerCCModePWM,
      .filter = false,
      .prsInput = false,
      //coist: true==>counter stop high,false counter stop low
      .coist = (sg_pwm_cfg_tbl.pwm_info[ch_id].idle_level) ? false:true,// : false,
      // Positive or negative, invert = 0:Positive, 1:negative
      .outInvert = !(info->polarity),
    };
    if (v_timer_sn == PWM_TIMER0_ID) {
      TIMER_InitCC(TIMER0, v_cct_sn, &timerCCInit);
      TIMER_Enable(TIMER0, TRUE);
    }
    else if (v_timer_sn == PWM_TIMER1_ID) {
      TIMER_InitCC(TIMER1, v_cct_sn, &timerCCInit);
      TIMER_Enable(TIMER1, TRUE);
    }
  }
  
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.frequency != info->frequency) {
    sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period = sg_ticksPerPeriod;
    if (v_timer_sn == PWM_TIMER0_ID) {
      //TIMER_TopSet(TIMER0, v_ticksPerPeriod32-1);
      TIMER_TopBufSet(TIMER0, sg_ticksPerPeriod-1);
    }
    else if (v_timer_sn == PWM_TIMER1_ID) {
      //TIMER_TopSet(TIMER1, v_ticksPerPeriod32-1);
      TIMER_TopBufSet(TIMER1, sg_ticksPerPeriod-1);
    }
  }
  
  
  if (sg_pwm_cfg_tbl.pwm_info[ch_id].cfg.duty != info->duty) {
    // update the Settings PWM parameters effective
    if (v_timer_sn == PWM_TIMER0_ID) {
      TIMER_CompareBufSet(TIMER0, v_cct_sn,
                          info->duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
    //   TIMER_CompareSet(TIMER0, v_cct_sn,
    //                    info->duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
    }
    
    if (v_timer_sn == PWM_TIMER1_ID) {
      TIMER_CompareBufSet(TIMER1, v_cct_sn,
                          info->duty*(sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
    //   TIMER_CompareSet(TIMER1, v_cct_sn,
    //                    info->duty * (sg_pwm_cfg_tbl.pwm_info[ch_id].tick_per_period) / sg_pwm_cfg_tbl.pwm_info[ch_id].precision);
    }
  }
  tkl_system_memcpy(&sg_pwm_cfg_tbl.pwm_info[ch_id].cfg, info, sizeof(TUYA_PWM_BASE_CFG_T));
  
  return OPRT_OK;
}

OPERATE_RET tkl_pwm_info_get(TUYA_PWM_NUM_E ch_id, TUYA_PWM_BASE_CFG_T *info)
{
  if (ch_id >= PWM_MAX_CHANNEL || NULL == info) {
    return OPRT_INVALID_PARM;
  }
  
  tkl_system_memcpy(info, &sg_pwm_cfg_tbl.pwm_info[ch_id].cfg, sizeof(TUYA_PWM_BASE_CFG_T));
  
  return OPRT_OK;
}
/******************************************************************************/
/**                              EOF                                         **/
/******************************************************************************/
