/**
* @file tkl_uart.c
* @brief This is tuya tkl uart src file
* @version 0.1
* @date 2021-08-06
*
* @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
*
*/
#include "em_cmu.h"
#include "em_usart.h"
#include "gpiointerrupt.h"
#if defined (TUYA_RUNTIME_DBG)
#include "tkl_platform_types.h"
#endif

#include "tkl_memory.h"
#include "tkl_gpio.h"
#include "tkl_uart.h"
/******************************************************************************/
/**                  module description following                            **/
/******************************************************************************/
/* Main Features below:


*/

/******************************************************************************/
/**                  internal macro definition following                     **/
/******************************************************************************/

// uart default gpio id
#define UART_MAX_NUM        2
#define UART0_ID            0
#define UART0_TX_GPIO_ID    5  //PA5
#define UART0_RX_GPIO_ID    6  //PA6
#define UART1_ID            1
#define UART1_TX_GPIO_ID    0  //PA0
#define UART1_RX_GPIO_ID    1  //PA1


#define USART_INITASYNC_UART_DEFAULT                                        \
    {                                                                       \
        usartEnable, /* Enable RX/TX when init completed. */                \
        0,       /* Use current configured reference clock for configuring \
                    baudrate. */                                           \
        115200,  /* 115200 bits/s. */                                      \
        usartOVS16,     /* 16x oversampling. */                            \
        usartDatabits8, /* 8 databits. */                                  \
        usartNoParity,  /* No parity. */                                   \
        usartStopbits1, /* 1 stopbit. */                                   \
        false,          /* Do not disable majority vote. */                \
        false,          /* Not USART PRS input mode. */                    \
        usartPrsRxCh0,  /* PRS channel 0. */                               \
        false,          /* Auto CS functionality enable/disable switch */  \
        0,              /* Auto CS Hold cycles */                          \
        0,              /* Auto CS Setup cycles */                         \
        usartHwFlowControlNone /* No HW flow control */                    \
    }
/******************************************************************************/
/**                  internal type definition following                      **/
/******************************************************************************/

typedef union {
    UINT8_T all;
    struct {
        UINT8_T is_uart_init : 1;
        UINT8_T is_pin_init  : 1;
        UINT8_T is_int_send  : 1;
        UINT8_T reserved     : 5;
    } area;
} __UART_STATE_T;

typedef struct {
    UINT8_T tx_local;  // not used in MG21 platform
    UINT8_T rx_local;  // not used in MG21 platform
    UINT32_T tx_pin;
    UINT32_T rx_pin;
} __UART_PIN_T;

typedef struct {
    UINT32_T uart_id;
    TUYA_UART_BASE_CFG_T uart_cfg;
    __UART_PIN_T     pin_cfg;
    __UART_STATE_T   state;

    TUYA_UART_IRQ_CB recv_cb;
    TUYA_UART_IRQ_CB send_cb;
    UINT8_T          data;
} __UART_MGR_T;

typedef struct {
    USART_TypeDef* v_port;
    CMU_Clock_TypeDef uartClock;
    UINT32_T baudRate;
    USART_Databits_TypeDef databits;
    USART_Parity_TypeDef parity;
    USART_Stopbits_TypeDef stopbits;
    USART_HwFlowControl_TypeDef FlowCtrl;
} __NATIVE_UART_CFG_T;
/******************************************************************************/
/**                  global veriables definition following                   **/
/******************************************************************************/
// default uart_port-gpio map
STATIC struct {
    UINT32_T rx_id;
    UINT32_T tx_id;
} sg_uart_map[UART_MAX_NUM] = {
    {UART0_RX_GPIO_ID, UART0_TX_GPIO_ID},
    {UART1_RX_GPIO_ID, UART1_TX_GPIO_ID},
};

STATIC __UART_MGR_T sg_uart_mgr_tbl[UART_MAX_NUM];

extern UINT8_T __get_gpio_pin_num(UINT32_T pin_id);
extern UINT8_T __get_gpio_port_num(UINT32_T pin_id);

/******************************************************************************/
/**                  functions definition following                          **/
/******************************************************************************/

STATIC VOID_T __uart_receive_callback(UINT32_T v_port, UCHAR_T value)
{
    if (!sg_uart_mgr_tbl[v_port].state.area.is_uart_init) {
        return;
    }
    if (sg_uart_mgr_tbl[v_port].recv_cb) {
        sg_uart_mgr_tbl[v_port].data = value;
        sg_uart_mgr_tbl[v_port].recv_cb(v_port);
    }
}

STATIC VOID_T __uart_send_callback(UINT32_T v_port)
{
    if (!sg_uart_mgr_tbl[v_port].state.area.is_uart_init ||
        !sg_uart_mgr_tbl[v_port].state.area.is_int_send) {
        return;
    }

    if (sg_uart_mgr_tbl[v_port].send_cb) {
        sg_uart_mgr_tbl[v_port].send_cb(v_port);
    }
}

STATIC VOID_T __uart_default_pin_set(UINT32_T v_port,
                                     __UART_PIN_T* uart_pin)
{
    uart_pin->tx_local = 0;
    uart_pin->tx_pin = sg_uart_map[v_port].tx_id;
    uart_pin->rx_local = 0;
    uart_pin->rx_pin = sg_uart_map[v_port].rx_id;
}

STATIC VOID_T __uart_struct_data_conversion(__UART_MGR_T* mgr_cfg,
                                            __NATIVE_UART_CFG_T* native_cfg)
{
    if (NULL == mgr_cfg || NULL == native_cfg) {
        return;
    }

    native_cfg->v_port = USART0;
    native_cfg->uartClock = cmuClock_USART0;
    // NOTE: not support hardware flow control
    native_cfg->FlowCtrl = usartHwFlowControlNone;
    native_cfg->baudRate = (UINT32_T)(mgr_cfg->uart_cfg.baudrate);

    if (mgr_cfg->uart_id == UART1_ID) {
        native_cfg->v_port = USART1;
        native_cfg->uartClock = cmuClock_USART1;
    }

    // databits
    switch (mgr_cfg->uart_cfg.databits) {
        case TUYA_UART_DATA_LEN_5BIT:
            native_cfg->databits = usartDatabits5;
            break;
        case TUYA_UART_DATA_LEN_6BIT:
            native_cfg->databits = usartDatabits6;
            break;
        case TUYA_UART_DATA_LEN_7BIT:
            native_cfg->databits = usartDatabits7;
            break;
        case TUYA_UART_DATA_LEN_8BIT:
            native_cfg->databits = usartDatabits8;
            break;
        default:
            native_cfg->databits = usartDatabits8;
            break;
    }

    // parity
    switch (mgr_cfg->uart_cfg.parity) {
        case TUYA_UART_PARITY_TYPE_NONE:
            native_cfg->parity = usartNoParity;
            break;
        case TUYA_UART_PARITY_TYPE_ODD:
            native_cfg->parity = usartOddParity;
            break;
        case TUYA_UART_PARITY_TYPE_EVEN:
            native_cfg->parity = usartEvenParity;
            break;
        default:
            native_cfg->parity = usartNoParity;
            break;
    }

    // stopbits
    switch (mgr_cfg->uart_cfg.stopbits) {
        case TUYA_UART_STOP_LEN_1BIT:
            native_cfg->stopbits = usartStopbits1;
            break;
        case TUYA_UART_STOP_LEN_1_5BIT1:
            native_cfg->stopbits = usartStopbits1p5;
            break;
        case TUYA_UART_STOP_LEN_2BIT:
            native_cfg->stopbits = usartStopbits2;
            break;
        default:
            native_cfg->stopbits = usartStopbits1;
            break;
    }
}


STATIC VOID_T __uart_init_base(__UART_PIN_T* pin,
                               __NATIVE_UART_CFG_T* cfg)
{
    UINT8_T uart_num = 0;
    USART_InitAsync_TypeDef uartInit = USART_INITASYNC_UART_DEFAULT;

    uartInit.baudrate = cfg->baudRate;
    uartInit.databits = cfg->databits;
    uartInit.stopbits = cfg->stopbits;
    uartInit.parity = cfg->parity;
    uartInit.hwFlowControl = cfg->FlowCtrl;
    if (cfg->v_port == USART1) {
        uart_num = 1;
    }

/* Enable GPIO in CMU */
#if !defined(_SILICON_LABS_32B_SERIES_2)
    CMU_ClockEnable(cmuClock_HFPER, TRUE);
#endif  //! defined(_SILICON_LABS_32B_SERIES_2)
    CMU_ClockEnable(cmuClock_GPIO, TRUE);
    CMU_ClockEnable(cfg->uartClock, TRUE);

    /* Reseting and initializing LEUART */
    USART_Reset(cfg->v_port);
    USART_InitAsync(cfg->v_port, &uartInit);

#if defined(EFR32MG13P)
    // Set up RX pin
    cfg->v_port->ROUTELOC0 =
        (cfg->v_port->ROUTELOC0 & (~_USART_ROUTELOC0_RXLOC_MASK)) |
        ((pin->rx.local) << _USART_ROUTELOC0_RXLOC_SHIFT);
    cfg->v_port->ROUTEPEN = cfg->v_port->ROUTEPEN | USART_ROUTEPEN_RXPEN;

    // Set up TX pin
    cfg->v_port->ROUTELOC0 =
        (cfg->v_port->ROUTELOC0 & (~_USART_ROUTELOC0_TXLOC_MASK)) |
        ((pin->tx.local) << _USART_ROUTELOC0_TXLOC_SHIFT);
    cfg->v_port->ROUTEPEN = cfg->v_port->ROUTEPEN | USART_ROUTEPEN_TXPEN;
#elif defined(GPIO_USART_ROUTEEN_TXPEN)
    GPIO->USARTROUTE[uart_num].TXROUTE =
        ((__get_gpio_port_num(pin->tx_pin)) << _GPIO_USART_TXROUTE_PORT_SHIFT) |
        ((__get_gpio_pin_num(pin->tx_pin)) << _GPIO_USART_TXROUTE_PIN_SHIFT);
    GPIO->USARTROUTE[uart_num].RXROUTE =
        ((__get_gpio_port_num(pin->rx_pin)) << _GPIO_USART_RXROUTE_PORT_SHIFT) |
        ((__get_gpio_pin_num(pin->rx_pin)) << _GPIO_USART_RXROUTE_PIN_SHIFT);
    GPIO->USARTROUTE[uart_num].ROUTEEN =
            GPIO_USART_ROUTEEN_TXPEN | GPIO_USART_ROUTEEN_RXPEN;
#endif
    GPIO_PinModeSet((GPIO_Port_TypeDef)(__get_gpio_port_num(pin->tx_pin)), /* Port */
                    (__get_gpio_pin_num(pin->tx_pin)),                     /* Port number */
                    gpioModePushPull, /* Pin mode is set to input only, with pull direction given bellow */
                    1);               /* Pull direction is set to pull-up */
    GPIO_PinModeSet((GPIO_Port_TypeDef)(__get_gpio_port_num(pin->rx_pin)), /* Port */
                    (__get_gpio_pin_num(pin->rx_pin)),                     /* Port number */
                    gpioModeInputPull, /* Pin mode is set to input only, with pull direction given bellow */
                    1);                /* Pull direction is set to pull-up */
    USART_IntClear(cfg->v_port, ~0x0);
    USART_Enable(cfg->v_port, usartEnable);
}

STATIC VOID_T __uart_deinit_base(__UART_PIN_T* pin,
                                 __NATIVE_UART_CFG_T* cfg)
{
    UINT8_T uart_num = 0;
    USART_TypeDef* hal_port = USART0;

    CMU_ClockEnable(cfg->uartClock, FALSE);
#if defined(GPIO_USART_ROUTEEN_TXPEN)
    GPIO->USARTROUTE[uart_num].ROUTEEN &=
        ~(GPIO_USART_ROUTEEN_TXPEN | GPIO_USART_ROUTEEN_RXPEN);
    GPIO->USARTROUTE[uart_num].TXROUTE &=
        ~(((__get_gpio_port_num(pin->tx_pin)) << _GPIO_USART_TXROUTE_PORT_SHIFT) |
          ((__get_gpio_pin_num(pin->tx_pin)) << _GPIO_USART_TXROUTE_PIN_SHIFT));
    GPIO->USARTROUTE[uart_num].RXROUTE &=
        ~(((__get_gpio_port_num(pin->rx_pin)) << _GPIO_USART_RXROUTE_PORT_SHIFT) |
          ((__get_gpio_pin_num(pin->rx_pin)) << _GPIO_USART_RXROUTE_PIN_SHIFT));
#endif

    USART_IntDisable(hal_port, USART_IF_RXDATAV);
    USART_IntClear(hal_port, USART_IF_RXDATAV);
    USART_Enable(hal_port, usartDisable);

    GPIO_PinModeSet((GPIO_Port_TypeDef)(__get_gpio_port_num(pin->tx_pin)), /* Port */
                    (__get_gpio_pin_num(pin->tx_pin)),                     /* Port number */
                    gpioModeInput, /* Pin mode is set to input only, with pull direction given bellow */
                    1);            /* Pull direction is set to pull-up */
    GPIO_PinModeSet((GPIO_Port_TypeDef)(__get_gpio_port_num(pin->rx_pin)), /* Port */
                    (__get_gpio_pin_num(pin->rx_pin)),                     /* Port number */
                    gpioModeInput, /* Pin mode is set to input only, with pull direction given bellow */
                    1);            /* Pull direction is set to pull-up */
}

OPERATE_RET tkl_uart_init(TUYA_UART_NUM_E port_id, TUYA_UART_BASE_CFG_T *cfg)
{
    __NATIVE_UART_CFG_T v_native_cfg;

    if (port_id >= UART_MAX_NUM || cfg == NULL) {
        return OPRT_INVALID_PARM;
    }
    
    sg_uart_mgr_tbl[port_id].state.area.is_uart_init = FALSE;
    sg_uart_mgr_tbl[port_id].uart_id = port_id;
    if (!sg_uart_mgr_tbl[port_id].state.area.is_pin_init) {
        __uart_default_pin_set(port_id, &(sg_uart_mgr_tbl[port_id].pin_cfg));
        sg_uart_mgr_tbl[port_id].state.area.is_pin_init = TRUE;
    }

    tkl_system_memcpy(&(sg_uart_mgr_tbl[port_id].uart_cfg), cfg, sizeof(TUYA_UART_BASE_CFG_T));

    __uart_struct_data_conversion(&sg_uart_mgr_tbl[port_id], &v_native_cfg);
    __uart_init_base(&(sg_uart_mgr_tbl[port_id].pin_cfg), &v_native_cfg);

    sg_uart_mgr_tbl[port_id].state.area.is_uart_init = TRUE;

    return OPRT_OK;
}

OPERATE_RET tkl_uart_deinit(TUYA_UART_NUM_E port_id)
{
    __NATIVE_UART_CFG_T native_cfg;

    if (port_id >= UART_MAX_NUM) {
        return OPRT_INVALID_PARM;
    }
    if (!sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return OPRT_OK;
    }

    sg_uart_mgr_tbl[port_id].state.all = 0;
    __uart_struct_data_conversion(&sg_uart_mgr_tbl[port_id], &native_cfg);
    __uart_deinit_base(&(sg_uart_mgr_tbl[port_id].pin_cfg), &native_cfg);

    return OPRT_OK;
}

OPERATE_RET tkl_uart_mapping_to_gpio(TUYA_UART_NUM_E port_id, TUYA_GPIO_NUM_E tx_gpio_id, TUYA_GPIO_NUM_E rx_gpio_id)
{
    if (port_id >= UART_MAX_NUM) {
        return OPRT_INVALID_PARM;
    }

    sg_uart_map[port_id].tx_id = (UINT32_T)tx_gpio_id;
    sg_uart_map[port_id].rx_id = (UINT32_T)rx_gpio_id;
    
    return OPRT_OK;
}


INT_T tkl_uart_write(TUYA_UART_NUM_E port_id, VOID_T *buff, UINT16_T len)
{
    UINT16_T send_len = len;
    UINT8_T* data = (UINT8_T*)buff;
    USART_TypeDef* v_uart_id = USART0;

    if (NULL == buff || len == 0) {
        return OPRT_INVALID_PARM;
    }
    if (port_id >= UART_MAX_NUM ||
        !sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return OPRT_INVALID_PARM;
    }

    if (port_id == UART1_ID) {
        v_uart_id = USART1;
    }

    if (!sg_uart_mgr_tbl[port_id].state.area.is_int_send) {
        do {
            USART_Tx(v_uart_id, *data++);
            send_len--;
        } while (send_len > 0);
    } 
    else {
        tkl_uart_set_tx_int(port_id, TRUE);
        USART_Tx(v_uart_id, data[0]);
    }
    return len;
}


VOID_T tkl_uart_rx_irq_cb_reg(TUYA_UART_NUM_E port_id, TUYA_UART_IRQ_CB rx_cb)
{
    USART_TypeDef* hal_port = USART0;
    IRQn_Type uart_irq = USART0_RX_IRQn;

    if (port_id >= UART_MAX_NUM || rx_cb == NULL ||
        !sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return ;
    }

    sg_uart_mgr_tbl[port_id].recv_cb = rx_cb;
    if (port_id == UART1_ID) {
        hal_port = USART1;
        uart_irq = USART1_RX_IRQn;
    }

    NVIC_SetPriority(uart_irq, 3);
    NVIC_ClearPendingIRQ(uart_irq);
    USART_IntClear(hal_port, ~0x0);

    USART_IntEnable(hal_port, USART_IF_RXDATAV);
    NVIC_EnableIRQ(uart_irq);
}


VOID_T tkl_uart_tx_irq_cb_reg(TUYA_UART_NUM_E port_id, TUYA_UART_IRQ_CB tx_cb)
{
    USART_TypeDef* hal_port = USART0;
    IRQn_Type uart_irq = USART0_TX_IRQn;

    if (port_id >= UART_MAX_NUM || tx_cb == NULL ||
        !sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return ;
    }

    sg_uart_mgr_tbl[port_id].send_cb = tx_cb;
    sg_uart_mgr_tbl[port_id].state.area.is_int_send = 1;

    if (port_id == UART1_ID) {
        hal_port = USART1;
        uart_irq = USART1_TX_IRQn;
    }

    NVIC_SetPriority(uart_irq, 3);
    NVIC_ClearPendingIRQ(uart_irq);
    USART_IntClear(hal_port, ~0x0);

    NVIC_EnableIRQ(uart_irq);
}


INT_T tkl_uart_read(TUYA_UART_NUM_E port_id, VOID_T *buff, UINT16_T len)
{
    UINT8_T* data = (UINT8_T*)buff;
    
    if (data == NULL || len == 0) {
        return OPRT_INVALID_PARM;
    }
    if (port_id >= UART_MAX_NUM ||
        !sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return OPRT_INVALID_PARM;
    }
    *data = sg_uart_mgr_tbl[port_id].data;
    
    return 1;
}


OPERATE_RET tkl_uart_set_tx_int(TUYA_UART_NUM_E port_id, BOOL_T enable)
{
    USART_TypeDef* hal_port = USART0;

    if (port_id >= UART_MAX_NUM || 
        !sg_uart_mgr_tbl[port_id].state.area.is_uart_init) {
        return OPRT_INVALID_PARM;
    }

    if (port_id == UART1_ID) {
        hal_port = USART1;
    }
    if (enable) {
        USART_IntEnable(hal_port, USART_IF_TXC);
    } else {
        USART_IntDisable(hal_port, USART_IF_TXC);
    }

    return OPRT_OK;
}


OPERATE_RET tkl_uart_set_rx_flowctrl(TUYA_UART_NUM_E port_id, BOOL_T enable)
{
    return OPRT_NOT_SUPPORTED;
}

void USART0_RX_IRQHandler(void) 
{
    UINT8_T rx_data;
    if (USART0->STATUS & USART_STATUS_RXDATAV) {
        rx_data = USART_Rx(USART0);
        __uart_receive_callback(UART0_ID, rx_data);
    }
}
void USART1_RX_IRQHandler(void)
{
    UINT8_T rx_data;
    if (USART1->STATUS & USART_STATUS_RXDATAV) {
        rx_data = USART_Rx(USART1);
        __uart_receive_callback(UART1_ID, rx_data);
    }
}
void USART0_TX_IRQHandler(void)
{
    if (USART0->STATUS & USART_STATUS_TXC) {
        __uart_send_callback(UART0_ID);
    }
}

void USART1_TX_IRQHandler(void)
{
    if (USART1->STATUS & USART_STATUS_TXC) {
        __uart_send_callback(UART1_ID);
    }
}

#if defined (TUYA_RUNTIME_DBG)
VOID_T tkl_dbg_printf(const CHAR_T *formatString, ...)
{
    COM_Port_t hal_port = COM_USART0;
    USHORT_T v_port = (USHORT_T)UART0_ID;

    if (!sg_uart_mgr_tbl[v_port].state.area.is_uart_init) {
        TUYA_UART_BASE_CFG_T cfg = {
            .baudrate = 115200,
            .parity = TUYA_UART_PARITY_TYPE_NONE,
            .databits = TUYA_UART_DATA_LEN_8BIT,
            .stopbits = TUYA_UART_STOP_LEN_1BIT,
            .flowctrl = TUYA_UART_FLOWCTRL_NONE,
        };
        tkl_uart_init(UART0_ID, &cfg);
    }

    va_list ap;
    va_start(ap, formatString);
    emberSerialPrintfVarArg(hal_port, formatString, ap);
    va_end(ap);
}

VOID_T tkl_dbg_printfln(const CHAR_T *formatString, ...)
{
    COM_Port_t hal_port = COM_USART0;
    USHORT_T v_port = (USHORT_T)UART0_ID;

    if (!sg_uart_mgr_tbl[v_port].state.area.is_uart_init) {
        TUYA_UART_BASE_CFG_T cfg = {
            .baudrate = 115200,
            .parity = TUYA_UART_PARITY_TYPE_NONE,
            .databits = TUYA_UART_DATA_LEN_8BIT,
            .stopbits = TUYA_UART_STOP_LEN_1BIT,
            .flowctrl = TUYA_UART_FLOWCTRL_NONE,
        };
        tkl_uart_init(UART0_ID, &cfg);
    }

    CHAR_T* buff= "\r\n";
    va_list ap;
    va_start(ap, formatString);
    emberSerialPrintfVarArg(hal_port, formatString, ap);
    va_end(ap);
    tkl_uart_write(UART0_ID, (VOID_T *)buff, 2);
}

// rewrite sdk function
Ecode_t COM_WriteData(COM_Port_t port, uint8_t *data, uint8_t length)
{
    UINT32_T id = UART0_ID;
    
    if(port == COM_USART2) {
        id = UART1_ID;
    }
    if(tkl_uart_write(id, (VOID_T *)data, (UINT16_T)length) > 0) {
        return 0;
    }
    return -1;
}
#endif

/******************************************************************************/
/**                              EOF                                         **/
/******************************************************************************/
