/**
* @file tkl_watchdog.c
* @brief This is tuya tkl watchdog src file
* @version 0.1
* @date 2021-08-06
*
* @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
*
*/
#include "em_cmu.h"
#include "em_wdog.h"

#include "tkl_watchdog.h"
/******************************************************************************/
/**                  module description following                            **/
/******************************************************************************/
/* Main Features below:


*/
/******************************************************************************/
/**                  internal macro definition following                     **/
/******************************************************************************/

/******************************************************************************/
/**                  internal type definition following                      **/
/******************************************************************************/

/******************************************************************************/
/**                  internal function declaration following                 **/
/******************************************************************************/

/******************************************************************************/
/**                  global veriables definition following                   **/
/******************************************************************************/

/******************************************************************************/
/**                  functions definition following                          **/
/******************************************************************************/
STATIC WDOG_PeriodSel_TypeDef __get_wdt_peroid_param(UINT32_T interval_ms,
                                                     UINT32_T* real_ms) 
{
    *real_ms = 16385;
    WDOG_PeriodSel_TypeDef PeriodSel = wdogPeriod_16k;

    if (interval_ms <= 1025) {
        *real_ms = 1025;
        return wdogPeriod_1k;
    } else if (interval_ms <= 2049) {
        *real_ms = 2049;
        return wdogPeriod_2k;
    } else if (interval_ms <= 4097) {
        *real_ms = 4097;
        return wdogPeriod_4k;
    } else if (interval_ms <= 8193) {
        *real_ms = 8193;
        return wdogPeriod_8k;
    } 
    return PeriodSel;
}
/**
 * @brief watchdog init
 *
 * @param[in] cfg: watchdog config
 *
 * @return 0, init error; >0 : the actually watchdog interval
 */
UINT32_T tkl_watchdog_init(TUYA_WDOG_BASE_CFG_T* cfg)
{
    if (NULL == cfg) {
        return OPRT_COM_ERROR;
    }
    UINT32_T v_real_ms = 0;
    WDOG_Init_TypeDef v_native_cfg = {
        .enable = true,    /* Start watchdog when init done */
        .debugRun = false, /* WDOG not counting during debug halt */
        .em2Run = true,    /* WDOG counting when in EM2 */
        .em3Run = true,    /* WDOG counting when in EM3 */
        .em4Block = false, /* EM4 can be entered */
        .lock = false,     /* Do not lock WDOG configuration (if locked, reset
                              needed to unlock) */
        .perSel = __get_wdt_peroid_param(cfg->interval_ms, &v_real_ms),
    };

    CMU_ClockSelectSet(cmuClock_WDOG0, cmuSelect_ULFRCO);
    WDOG_Init(&v_native_cfg);

    return v_real_ms;
}

/**
 * @brief watchdog deinit
 *
 * @param[in] none
 *
 * @return OPRT_OK on success. Others on error, please refer to
 * tuya_error_code.h
 */
OPERATE_RET tkl_watchdog_deinit(VOID_T)
{
    WDOGn_Enable(DEFAULT_WDOG, FALSE);
    return OPRT_OK;
}

/**
 * @brief refresh watch dog
 *
 * @param[in] none
 *
 * @return OPRT_OK on success. Others on error, please refer to
 * tuya_error_code.h
 */
OPERATE_RET tkl_watchdog_refresh(VOID_T)
{
    WDOGn_Feed(DEFAULT_WDOG);
    return OPRT_OK;
}

/******************************************************************************/
/**                              EOF                                         **/
/******************************************************************************/
