/**
* @file tkl_memory.c
* @brief This is tuya tkl memory src file
* @version 0.1
* @date 2021-04-06
*
* @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
*
*/

#include <stdlib.h>
#include <string.h>

#include "tuya_mem_heap.h"

#include "tkl_system.h"
#include "tkl_memory.h"

STATIC UINT8_T is_init;
STATIC UINT_T irq_mask;
STATIC HEAP_HANDLE mem_handle;

#define TOTAL_HEAP_SIZE (10*1024UL)
STATIC UINT8_T gs_mem_heap[TOTAL_HEAP_SIZE];

TUYA_WEAK_ATTRIBUTE VOID_T tkl_memory_cfg(UINT8_T **out_buff, UINT32_T *out_buff_len)
{
    *out_buff = gs_mem_heap;
    *out_buff_len = TOTAL_HEAP_SIZE;
}


STATIC VOID_T __enter_critical(VOID_T)
{
    irq_mask = tkl_system_enter_critical();   
}
STATIC VOID_T __exit_critical(VOID_T)
{
    tkl_system_exit_critical(irq_mask);
}
STATIC VOID_T __printf(char* format, ...)
{

}

INT_T tkl_system_get_free_heap_size(VOID_T)
{
    return tuya_mem_heap_available(mem_handle);
}

VOID_T* tkl_system_malloc(CONST SIZE_T size)
{
    if(!is_init) {
        heap_context_t heap_context = {
            .enter_critical = __enter_critical,
            .exit_critical = __exit_critical,
            .dbg_output = __printf,
        };
        
        tuya_mem_heap_init(&heap_context);
        UINT8_T *buff;
        UINT32_T buff_len;
        tkl_memory_cfg(&buff, &buff_len);
        if(tuya_mem_heap_create((VOID_T*)buff, buff_len, &mem_handle)) {
            return NULL;
        }
        is_init = 1;
    }

    return tuya_mem_heap_malloc(mem_handle, size);
}


VOID_T tkl_system_free(VOID_T* ptr)
{
    tuya_mem_heap_free(mem_handle, ptr);
}


VOID_T *tkl_system_calloc(size_t nitems, size_t size)
{
    return tuya_mem_heap_calloc(mem_handle, nitems * size);
}


VOID_T *tkl_system_realloc(VOID_T* ptr, size_t size)
{
    return NULL;
}


VOID_T *tkl_system_memset(VOID_T* src, INT_T ch, SIZE_T n)
{
    return memset(src, ch, n);
}


VOID_T *tkl_system_memcpy(VOID_T* dst, CONST VOID_T* src, SIZE_T n)
{
    return memcpy(dst, src, n);
}

INT_T tkl_system_memcmp(CONST VOID_T *str1, CONST VOID_T *str2, SIZE_T n)
{
    return memcmp(str1, str2, n);
}
