/**
* @file tkl_system.c
* @brief This is tuya tkl system src file
* @version 0.1
* @date 2021-04-06
*
* @copyright Copyright 2021-2022 Tuya Inc. All Rights Reserved.
*
*/
#include "tkl_platform_types.h"
#include "tkl_memory.h"
#include "tkl_system.h"


VOID_T tkl_system_reset(VOID_T)
{
    NVIC_SystemReset();
}

INT_T tkl_system_get_random(UINT_T range)
{
    INT_T random = 0;

    if(range != 0) {
        random = ((INT_T)halCommonGetRandom() + halCommonGetRandom()) % range;
    }
    return random;
}

UINT_T tkl_system_enter_critical(VOID_T)
{
    return CORE_EnterAtomic();
}

VOID_T tkl_system_exit_critical(UINT_T irq_mask)
{
    CORE_ExitAtomic(irq_mask);
}

SYS_TICK_T tkl_system_get_tick_count(VOID_T)
{
    return halCommonGetInt32uMillisecondTick();
}

SYS_TIME_T tkl_system_get_millisecond(VOID_T)
{
    return halCommonGetInt32uMillisecondTick();
}

TUYA_RESET_REASON_E tkl_system_get_reset_reason(CHAR_T** describe)
{
    // TUYA_RESET_REASON_POWERON    = 0,  ///< Poweron reset type, supply voltage < power-on threshold (TY_RST_POWER_OFF)
    // TUYA_RESET_REASON_HW_WDOG    = 1,  ///< Hardware watchdog reset occurred (TY_RST_HARDWARE_WATCHDOG)
    // TUYA_RESET_REASON_FAULT      = 2,  ///< A access fault occurred (TY_RST_FATAL_EXCEPTION)
    // TUYA_RESET_REASON_SW_WDOG    = 3,  ///< Software watchdog reset occurred (TY_RST_SOFTWARE_WATCHDOG)
    // TUYA_RESET_REASON_SOFTWARE   = 4,  ///< Software triggered reset (TY_RST_SOFTWARE)
    // TUYA_RESET_REASON_DEEPSLEEP  = 5,  ///< Reset caused by entering deep sleep (TY_RST_DEEPSLEEP)
    // TUYA_RESET_REASON_EXTERNAL   = 6,  ///< External reset trigger        (TY_RST_HARDWARE)
    // TUYA_RESET_REASON_UNKNOWN    = 7,  ///< Underterminable cause
    // TUYA_RESET_REASON_FIB        = 8,  ///< Reset originated from the FIB bootloader
    // TUYA_RESET_REASON_BOOTLOADER = 8,  ///< Reset relates to an bootloader
    // TUYA_RESET_REASON_CRASH      = 10, ///< Software crash
    // TUYA_RESET_REASON_FLASH      = 11, ///< Flash failure cause reset
    // TUYA_RESET_REASON_FATAL      = 12, ///< A non-recoverable fatal error occurred
    // TUYA_RESET_REASON_BROWNOUT   = 13, ///< Brown out
    // TUYA_RESET_REASON_UNSUPPORT  = 0xFF,
    UINT8_T reset_reason = halGetResetInfo();
    switch(reset_reason) {
        case 0:
            reset_reason = TUYA_RESET_REASON_UNKNOWN;
        break;
        case 1:
            reset_reason = TUYA_RESET_REASON_FIB;
        break;
        case 2:
            reset_reason = TUYA_RESET_REASON_BOOTLOADER;
        break;
        case 3:
            reset_reason = TUYA_RESET_REASON_EXTERNAL;
        break;
        case 4:
            reset_reason = TUYA_RESET_REASON_POWERON;
        break;
        case 5:
            reset_reason = TUYA_RESET_REASON_HW_WDOG;
        break;
        case 6:
            reset_reason = TUYA_RESET_REASON_SOFTWARE;
        break;
        case 7:
            reset_reason = TUYA_RESET_REASON_CRASH;
        break;
        case 8:
            reset_reason = TUYA_RESET_REASON_FLASH;
        break;
        case 9:
            reset_reason = TUYA_RESET_REASON_FATAL;
        break;
        case 10:
            reset_reason = TUYA_RESET_REASON_FAULT;
        break;
        case 11:
            reset_reason = TUYA_RESET_REASON_BROWNOUT;
        break;
        default:
            reset_reason = TUYA_RESET_REASON_UNSUPPORT;
        break;
    }
    return (TUYA_RESET_REASON_E)reset_reason;
}

