/*
 * @Author: Deven
 * @Email: liming@tuya.com
 * @LastEditors: Please set LastEditors
 * @FileName: &file name&
 * @Description: 
 * @Copyright: HANGZHOU TUYA INFORMATION TECHNOLOGY CO.,LTD
 * @Company: http://www.tuya.com
 * @Date: 2021-04-27 15:28:14
 * @LastEditTime: 2021-12-16 16:35:21
 */
#include "tkl_memory.h"
#include "tkl_platform_types.h"
#include "tkl_network_manager.h"

#if defined (TUYA_RUNTIME_DBG)
#include "tkl_uart.h"
#define tkl_nwk_mgr_debug(...)    tkl_dbg_printf(__VA_ARGS__)
#else
#define tkl_nwk_mgr_debug(...)
#endif


#define TUYA_DEFAULT_RADIO_POWER        10
#define TUYA_MAX_RADIO_POWER_DB         19
#define TUYA_MIN_RADIO_POWER_DB         0

#define TUYA_DEFAULT_RADIO_POWER_DB     11
#define TUYA_CH26_MAX_RADIO_POWER_DB    10



typedef struct {
    INT8_T ch26_max_radio_power_db;
    UINT8_T is_set_policy;
    UINT8_T permit_join;
    TKL_JOIN_CFG_T   join;
    TKL_REJOIN_CFG_T rejoin;
    TKL_TCLINK_KEY_CB tc_link_key_cb;
    TKL_ZGAL_NWK_CHANGE_CB nwk_change_cb;
} NWK_MGR_T;

NWK_MGR_T g_nwk_mgr={
    .ch26_max_radio_power_db =TUYA_CH26_MAX_RADIO_POWER_DB,
};

extern EmAfZigbeeProNetwork emAfZigbeeProNetworks[];
extern const EmAfZigbeeProNetwork *emAfCurrentZigbeeProNetwork;

EmberEventControl tkl_network_state_change_delay_event_control;

VOID_T __network_scan_result_cb(TKL_SCAN_TYPE_E type, TKL_SCAN_STATUS_E status);


/** @brief Zigbee Key Establishment
 *
 * A callback to the application to notify it of the status of the request for a
 * Link Key.
 *
 * @param partner partner The IEEE address of the partner device.  Or all zeros
 * if the Key establishment failed.  Ver.: always
 * @param status The status of the key establishment.  Ver.: always
 */
void emberAfZigbeeKeyEstablishmentCallback(EmberEUI64 partner,
                                        EmberKeyStatus status)
{
    TKL_KEY_STATUS_E key_status;
    
    if(status == EMBER_TRUST_CENTER_LINK_KEY_ESTABLISHED) {
        key_status = TKL_TCLINK_KEY_ESTABLISHED;
    }
    else if(status == EMBER_KEY_TABLE_FULL ||
            status == EMBER_KEY_ESTABLISHMENT_TIMEOUT) {
        key_status = TKL_TCLINK_KEY_ESTABLISHED_FAILED;
    }
    else if(status == EMBER_TRUST_CENTER_IS_PRE_R21 ||
            status == EMBER_VERIFY_LINK_KEY_SUCCESS) {
        key_status = TKL_TCLINK_KEY_VERIFY_SUCCESS;
    }
    else if(status == EMBER_VERIFY_LINK_KEY_FAILURE) {
        key_status = TKL_TCLINK_KEY_VERIFY_FAILED;
    }
    
    if(g_nwk_mgr.tc_link_key_cb != NULL) {
        g_nwk_mgr.tc_link_key_cb(key_status);
    }
}

OPERATE_RET tkl_zg_ch26_power_max_db_set(INT8_T db)
{
    if(db >TUYA_MAX_RADIO_POWER_DB || db <TUYA_MIN_RADIO_POWER_DB) {
        return OPRT_INVALID_PARM;
    }
    g_nwk_mgr.ch26_max_radio_power_db = db;

    return OPRT_OK;
}

void tkl_network_state_change_delay_event_handler(void)
{
    emberEventControlSetInactive(tkl_network_state_change_delay_event_control);
    if(g_nwk_mgr.nwk_change_cb != NULL) {
        g_nwk_mgr.nwk_change_cb(tkl_zg_nwk_base_status_get());
    }
}
/** @brief Stack Status
 *
 * This function is called by the application framework from the stack status
 * handler.  This callbacks provides applications an opportunity to be notified
 * of changes to the stack status and take appropriate action.  The return code
 * from this callback is ignored by the framework.  The framework will always
 * process the stack status after the callback returns.
 *
 * @param status   Ver.: always
 */
bool emberAfStackStatusCallback(EmberStatus status)
{
/*
 * - ::EMBER_NETWORK_UP
 * - ::EMBER_NETWORK_DOWN
 * - ::EMBER_JOIN_FAILED
 * - ::EMBER_MOVE_FAILED
 * - ::EMBER_CANNOT_JOIN_AS_ROUTER
 * - ::EMBER_NODE_ID_CHANGED
 * - ::EMBER_PAN_ID_CHANGED
 * - ::EMBER_CHANNEL_CHANGED
 * - ::EMBER_NETWORK_OPENED
 * - ::EMBER_NETWORK_CLOSED
 * - ::EMBER_NO_BEACONS
 * - ::EMBER_RECEIVED_KEY_IN_THE_CLEAR
 * - ::EMBER_NO_NETWORK_KEY_RECEIVED
 * - ::EMBER_NO_LINK_KEY_RECEIVED
 * - ::EMBER_PRECONFIGURED_KEY_REQUIRED
*/  
    tkl_nwk_mgr_debug("----Stack Status 0x%x ----\r\n",status);
    emberEventControlSetDelayMS(tkl_network_state_change_delay_event_control, 250);

    return false;
}

// void emberAfPluginEndDeviceSupportStackStatusCallback(EmberStatus status)
// {
//     __network_scan_result_cb(TKL_SCAN_FOR_REJOIN, TKL_SCAN_SUCESS);
// }

/** @brief Complete
 *
 * This callback is fired when the Network Steering plugin is complete.
 *
 * @param status On success this will be set to EMBER_SUCCESS to indicate a
 * network was joined successfully. On failure this will be the status code of
 * the last join or scan attempt. Ver.: always
 * @param totalBeacons The total number of 802.15.4 beacons that were heard,
 * including beacons from different devices with the same PAN ID. Ver.: always
 * @param joinAttempts The number of join attempts that were made to get onto
 * an open Zigbee network. Ver.: always
 * @param finalState The finishing state of the network steering process. From
 * this, one is able to tell on which channel mask and with which key the
 * process was complete. Ver.: always
 */
void emberAfPluginNetworkSteeringCompleteCallback(EmberStatus status,
                                                  UINT8_T totalBeacons,
                                                  UINT8_T joinAttempts,
                                                  UINT8_T finalState)
{
    TKL_SCAN_STATUS_E scan_status;
    tkl_nwk_mgr_debug("Join network complete: status: 0x%X, bc num: %d, attemp %d, st %d\r\n", status, totalBeacons, joinAttempts, finalState);
    tkl_nwk_mgr_debug("channel %d, nwk status 0x%x\r\n",emberAfGetRadioChannel(), emberNetworkState());
    switch (status) {
        case EMBER_SUCCESS: {
            scan_status = TKL_SCAN_SUCESS;
            break;
        }
        case EMBER_NO_BEACONS: {
            scan_status = TKL_SCAN_NO_BEACONS;
            break;
        }
        case EMBER_NO_LINK_KEY_RECEIVED: {
            scan_status = TKL_SCAN_NO_LINK_KEY;
        }
        
        default: {
            scan_status = TKL_SCAN_UNKNOWN_FAIL;
            break;
        }
    }
    __network_scan_result_cb(TKL_SCAN_FOR_JOIN, scan_status);
}


STATIC UINT16_T __install_code_crc16_x25(UINT8_T *data, UINT16_T len)
{
    UINT8_T c15,bit;
    UINT16_T crc16_out = 0;
    UINT16_T crc16 = 0xFFFF;
    UINT16_T i = 0,j = 0;
    
    for (i = 0; i < len ; i++) {
        for (j = 0; j < 8; j++) {
             c15 = ((crc16 >> 15 & 1) == 1);
             bit = ((data[i] >> j & 1) == 1);
             crc16 <<= 1;
             if (c15 ^ bit) {
                 crc16 ^= 0x1021;
             }
        }
    }

    for(i = 0; i <16; i++) {
        if(crc16 & (1 << i)) {
            crc16_out |= 1<<(15 - i);
        }
    }

    return (crc16_out^0xFFFF);
}

STATIC VOID_T __network_scan_result_cb(TKL_SCAN_TYPE_E type, TKL_SCAN_STATUS_E status)
{
    TKL_SCAN_RESULT_T result;

    // the channel is valid after joined a network 
    result.channel = emberAfGetRadioChannel();
    result.type = type;
    result.result = status;

    if(type == TKL_SCAN_FOR_JOIN) {
        if(g_nwk_mgr.join.scan_complete_cb) {
            g_nwk_mgr.join.scan_complete_cb(&result);
        }
    }
    else {
        if(g_nwk_mgr.rejoin.scan_complete_cb) {
            g_nwk_mgr.rejoin.scan_complete_cb(&result);
        }
    }
}

ZG_SCAN_DURATION_E tkl_zg_inner_scan_duration_get(VOID_T)
{
    if(!g_nwk_mgr.is_set_policy) {
        return ZG_SCAN_DURATION_3;
    }
    
    return g_nwk_mgr.join.duration;
}


TKL_NWK_BASIC_STATE_E tkl_zg_nwk_base_status_get(VOID_T)
{
    return (TKL_NWK_BASIC_STATE_E)emberNetworkState();
}

OPERATE_RET tkl_zg_nwk_state_change_register(TKL_ZGAL_NWK_CHANGE_CB nwk_change_cb)
{
    if(nwk_change_cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_nwk_mgr.nwk_change_cb = nwk_change_cb;

    return OPRT_OK;
}

VOID_T tkl_zg_scan_policy_set(TKL_SCAN_TYPE_E type, TKL_SCAN_POLICY_T *policy)
{
    if(type == TKL_SCAN_FOR_JOIN) {
        g_nwk_mgr.is_set_policy = TRUE;
        g_nwk_mgr.join.duration = policy->scan.join.duration;
        g_nwk_mgr.join.link_key_mask = policy->scan.join.link_key_mask;
        g_nwk_mgr.join.channel_mask = (policy->scan.join.channel_mask & EMBER_ALL_802_15_4_CHANNELS_MASK);
        g_nwk_mgr.join.scan_complete_cb = policy->scan.join.scan_complete_cb;
        emAfPluginNetworkSteeringSetChannelMask(0x00000000, TRUE);
        emAfPluginNetworkSteeringSetChannelMask(g_nwk_mgr.join.channel_mask, FALSE);
    }
    else {
        g_nwk_mgr.rejoin.type = policy->scan.rejoin.type;
        g_nwk_mgr.rejoin.scan_complete_cb = policy->scan.rejoin.scan_complete_cb;
    }
}

OPERATE_RET tkl_zg_start_scan(TKL_SCAN_TYPE_E type)
{
    EmberStatus status;
    UINT_T channel_mask;
    UINT8_T secure_rejoin = TRUE;
    
    if(type == TKL_SCAN_FOR_JOIN) {
        status = emberAfPluginNetworkSteeringStart();
    }
    else if(type == TKL_SCAN_FOR_REJOIN) {
        if(g_nwk_mgr.rejoin.type == TKL_REJOIN_ON_CURRENT_CH) {
            channel_mask = 0;
        }
        else {
            channel_mask = EMBER_ALL_802_15_4_CHANNELS_MASK;
        }
        status = emberFindAndRejoinNetworkWithReason(secure_rejoin,
                                                    channel_mask,
                                                    EMBER_AF_REJOIN_DUE_TO_END_DEVICE_MOVE);
    }

    return (status == EMBER_SUCCESS)? OPRT_OK: OPRT_OS_ADAPTER_ZG_SCAN_FAILED;
}

OPERATE_RET tkl_zg_stop_scan(TKL_SCAN_TYPE_E type)
{
    EmberStatus status = EMBER_SUCCESS;
    
    if(type == TKL_SCAN_FOR_JOIN) {
        status = emberAfPluginNetworkSteeringStop();
    }
    else {
        // NOTE: rejoin not have stop
    }

    return (status == EMBER_SUCCESS)? OPRT_OK: OPRT_OS_ADAPTER_ZG_STOP_SCAN_FAILED;
}


OPERATE_RET tkl_zg_start_leave(VOID_T)
{
    EmberStatus status;

    status = emberLeaveNetwork();

    return (status == EMBER_SUCCESS)? OPRT_OK: OPRT_OS_ADAPTER_ZG_LEAVE_FAILED;
}

VOID_T tkl_zg_install_code_set(TKL_INSTALL_CODE_T *insall_code)
{
    tokTypeMfgInstallationCode config;
    tokTypeMfgInstallationCode pre_config = {0};

    halCommonGetToken(&pre_config, TOKEN_MFG_INSTALLATION_CODE);
    if(pre_config.flags == 0xFFFF) {
        config.crc = __install_code_crc16_x25(insall_code->code, 16);
        tkl_nwk_mgr_debug("install code crc 0x%02x", config.crc);
        tkl_system_memcpy(config.value, insall_code->code, 16);
        config.flags = 0x0006;
        halCommonSetToken(TOKEN_MFG_INSTALLATION_CODE, &config);
    }
}

VOID_T tkl_zg_install_code_get(TKL_INSTALL_CODE_T *insall_code)
{
    tokTypeMfgInstallationCode pre_config;

    tkl_system_memset(&pre_config, 0, sizeof(tokTypeMfgInstallationCode));
    halCommonGetToken(&pre_config, TOKEN_MFG_INSTALLATION_CODE);
    tkl_system_memcpy(insall_code->code, pre_config.value, 16);
}

BOOL_T tkl_zg_nwk_secret_key_info_get(TKL_SECRET_KEY_T *key_info)
{
    EmberKeyStruct entry;

    tkl_system_memset(&entry, 0, sizeof(EmberKeyStruct));
    // if not joined a network,it will be is distributedTestKey
    EmberStatus keystatus = emberGetKey(EMBER_TRUST_CENTER_LINK_KEY, &entry);
    if(keystatus == EMBER_SUCCESS) {
        tkl_system_memcpy(key_info->sec_tc_link_key, entry.key.contents, 16);
        // if not joined a network,it will be a random value.
        tkl_system_memset(&entry, 0, sizeof(EmberKeyStruct));
        keystatus = emberGetKey(EMBER_CURRENT_NETWORK_KEY, &entry);
        if(keystatus == EMBER_SUCCESS) {
            tkl_system_memcpy(key_info->nwk_key, entry.key.contents, 16);
            return TRUE;
        }
    }

    return FALSE;
}


OPERATE_RET tkl_zg_register_key_establish_cb(TKL_TCLINK_KEY_CB key_cb)
{
    if(key_cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_nwk_mgr.tc_link_key_cb = key_cb;

    return OPRT_OK;
}

VOID_T tkl_zg_enable_permit_join_after_joined(BOOL_T enable)
{
    g_nwk_mgr.permit_join = (UINT8_T)enable;
}


BOOL_T tkl_zg_inner_is_permit_join_after_joined(VOID_T)
{
    return (UINT8_T)g_nwk_mgr.permit_join;
}


VOID_T tkl_zg_nwk_base_info_get(TKL_NWK_BASIC_INFO_T *nwk_info)
{
    TKL_NWK_BASIC_STATE_E nwk_status;

    // if not joined a network, channel and tx power is not true value.
    nwk_info->radio_channel = emberAfGetRadioChannel();
    nwk_info->radio_power = emberGetRadioPower();
    emberAfGetEui64(nwk_info->mac_addr);
    
    nwk_status = tkl_zg_nwk_base_status_get();
    if(nwk_status == TKL_NODE_JOINED_NETWORK ||
        nwk_status == TKL_NODE_NO_PARENT) {
        nwk_info->nwk_addr = emberAfGetNodeId();
        nwk_info->panid = emberAfGetPanId();
        emberGetExtendedPanId(nwk_info->ext_panid);
    }
    else {
        nwk_info->panid = 0xFFFF;
        nwk_info->nwk_addr = 0xFFFF;
        tkl_system_memset(nwk_info->ext_panid, 0xFF, 8);
    }
}

VOID_T tkl_zg_radio_power_set(INT8_T tx_power)
{
    tokTypeStackNodeData tokInf;

    // if not joined a network can not save tx_power to token
    if(tx_power <= TUYA_MIN_RADIO_POWER_DB || tx_power > TUYA_MAX_RADIO_POWER_DB) {
        tx_power = TUYA_DEFAULT_RADIO_POWER_DB;
    }

    if(emberAfGetRadioChannel() == 26) {
        if(tx_power > g_nwk_mgr.ch26_max_radio_power_db) {
            tx_power = g_nwk_mgr.ch26_max_radio_power_db;
        }
    }

    emberSetRadioPower(tx_power);
    /*update the token tx power*/
    halCommonGetToken(&tokInf, TOKEN_STACK_NODE_DATA);
    if(tokInf.radioTxPower != tx_power) {
        tokInf.radioTxPower = tx_power;
        halCommonSetToken(TOKEN_STACK_NODE_DATA, &tokInf);
    }
    tkl_nwk_mgr_debug("get tx power %d\r\n", tokInf.radioTxPower);
}

INT8_T tkl_zg_radio_power_get(VOID_T)
{
    return emberGetRadioPower();
}

UINT16_T tkl_zg_nwk_address_get(VOID_T)
{
    return emberGetNodeId();
}

UINT16_T tkl_zg_nwk_panid_get(VOID_T)
{
    return emberGetPanId();
}

VOID_T tkl_zg_nwk_extend_panid_get(UINT8_T *extpanid)
{
    emberGetExtendedPanId(extpanid);
}

VOID_T tkl_zg_mac_address_get(UINT8_T *mac)
{
    emberAfGetEui64(mac);
    // emberAfGetEui64(mac);
    //MEMMOVE(mac, emberGetEui64(), EUI64_SIZE);
}

UINT8_T tkl_zg_radio_channel_get(VOID_T)
{
    return emberGetRadioChannel();
}

VOID_T tkl_zg_radio_channel_set(UINT8_T channel)
{
    emberSetRadioChannel(channel);
}

VOID_T tkl_zg_node_type_set(ZG_NODE_TYPE_E type)
{
    emAfZigbeeProNetworks[0].nodeType = type + 2;
}

ZG_NODE_TYPE_E tkl_zg_node_type_get(VOID_T)
{
    return (ZG_NODE_TYPE_E)(emAfZigbeeProNetworks[0].nodeType - 2);
}

ZG_NWK_TYPE_E tkl_zg_nwk_type_get(VOID_T)
{
    BOOL_T is_distribute_nwk;
    
    is_distribute_nwk = emAfPluginNetworkIsJoinedDistributedNetwork();
    
    return is_distribute_nwk==FALSE ? ZG_CENTRALIZED_NWK : ZG_DISTRIBUTE_NWK;
}



extern EmberAfDefinedEndpoint *emAfEndpoints;
extern UINT8_T emberEndpointCount;

STATIC TKL_EXT_ZG_INFO_T *g_ext_zg_info = NULL;
STATIC UINT8_T g_ext_zg_info_sums = 0;

BOOL_T tkl_zg_set_ext_zg_info(TKL_EXT_ZG_INFO_T *p_zg_info, UINT8_T sums)
{
    UINT16_T totol_size = sizeof(TKL_EXT_ZG_INFO_T)*sums;
    
    g_ext_zg_info = tkl_system_malloc(totol_size);
    g_ext_zg_info_sums = sums;
    
    tkl_system_memcpy(g_ext_zg_info, p_zg_info, totol_size);

    return TRUE;
}

UINT16_T tkl_zg_found_dev_id(UINT16_T profile_id, UINT8_T endpoint)
{
    UINT8_T i;

    for(i=0; i<g_ext_zg_info_sums; i++) {
        if(profile_id != g_ext_zg_info[i].profile_id) {
            continue;
        }
        if(endpoint != g_ext_zg_info[i].endpoint) {
            continue;
        }

        return g_ext_zg_info[i].zg_device_id;
    }

    for(i=0; i<emberEndpointCount; i++) {
        if(emAfEndpoints[i].endpoint == endpoint) {
            return emAfEndpoints[i].deviceId;
        }
    }

    return 0xFFFF;
}

UINT8_T tkl_zg_found_version(UINT16_T profile_id, UINT8_T endpoint)
{
    UINT8_T i;

    for(i=0; i<g_ext_zg_info_sums; i++) {
        if(profile_id != g_ext_zg_info[i].profile_id) {
            continue;
        }
        if(endpoint != g_ext_zg_info[i].endpoint) {
            continue;
        }

        return g_ext_zg_info[i].version;
    }

    for(i=0; i<emberEndpointCount; i++) {
        if(emAfEndpoints[i].endpoint == endpoint) {
            return emAfEndpoints[i].deviceVersion;
        }
    }

    return 0xFF;
}