/*
 * @Author: Deven
 * @Email: liming@tuya.com
 * @LastEditors: Please set LastEditors
 * @FileName: &file name&
 * @Description: 
 * @Copyright: HANGZHOU TUYA INFORMATION TECHNOLOGY CO.,LTD
 * @Company: http://www.tuya.com
 * @Date: 2021-04-27 15:28:14
 * @LastEditTime: 2022-02-09 10:29:21
 */
#include "tkl_platform_types.h"
#include "tkl_memory.h"
#include "tkl_system.h"
#include "tkl_network_manager.h"
#include "tkl_network_recovery.h"

#if defined (TUYA_RUNTIME_DBG)
#include "tkl_uart.h"
#define tkl_nwk_reco_debug(...)     tkl_dbg_printf(__VA_ARGS__)
#else
#define tkl_nwk_reco_debug(...)
#endif

BOOL_T g_disable_recovery;

EmberEventControl tkl_network_recovery_reset_event_control;

extern VOID_T tkl_zg_nwk_recovery_status_callback(TKL_RECOVERY_STATUS_E status);

STATIC BOOL_T __network_security_set(UINT8_T *tcl_key, UINT8_T *nwk_key, UINT8_T *tc_eui64)
{
    EmberStatus status;
    EmberInitialSecurityState state = {0};
    state.bitmask = (EMBER_HAVE_PRECONFIGURED_KEY
                    | EMBER_HAVE_NETWORK_KEY
                    | EMBER_HAVE_TRUST_CENTER_EUI64
                    | EMBER_NO_FRAME_COUNTER_RESET
                    );

    MEMCOPY(emberKeyContents(&(state.preconfiguredKey)), tcl_key, EMBER_ENCRYPTION_KEY_SIZE);
    MEMCOPY(emberKeyContents(&(state.networkKey)), nwk_key, EMBER_ENCRYPTION_KEY_SIZE);
    MEMCOPY(state.preconfiguredTrustCenterEui64, tc_eui64, EUI64_SIZE);
    state.networkKeySequenceNumber = 0;

    status = emberSetInitialSecurityState(&state);
    if (status != EMBER_SUCCESS) {
        tkl_nwk_reco_debug("Error: could not setup security: 0x%x\r\n", status);
        return FALSE;
    }
    
    return TRUE; 
}

STATIC BOOL_T __network_info_recovery(EmberNetworkParameters *nwk_para, EmberNodeType node_type, UINT16_T nwk_addr)
{
    EmberStatus status;

    if(nwk_addr != 0xFFFF) {
        status = emberSetNodeId(nwk_addr);
        if (EMBER_SUCCESS != status) {
            tkl_nwk_reco_debug("Error: set node id: 0x%x\r\n",status);
            return FALSE;
        }
    }

    status = emberJoinNetwork(node_type, nwk_para);
    if (EMBER_SUCCESS != status) {
        tkl_nwk_reco_debug("Error: could not join: 0x%x\r\n",status);
        return FALSE;
    }
    
    return TRUE;
}

void tkl_network_recovery_reset_event_handler(void)
{
    EmberNetworkStatus nwk_status;
    
    emberEventControlSetInactive(tkl_network_recovery_reset_event_control);
    nwk_status = emberNetworkState();
    tkl_nwk_reco_debug("recovery reset:nwk status 0x%x\r\n", nwk_status);
    // prevent the problem of equipment restart and continuous distribution network caused by network recovery failure
    if(nwk_status == EMBER_JOINED_NETWORK || 
       nwk_status == EMBER_JOINED_NETWORK_NO_PARENT) {
        tkl_zg_nwk_recovery_status_callback(NWK_RECOVERY_SUCESS);
        tkl_system_reset();
    }
    else {
        tkl_zg_nwk_recovery_status_callback(NWK_RECOVERY_FAILED);
    }
}


VOID_T tkl_zg_nwk_recovery_init(VOID_T)
{
}

VOID_T tkl_zg_nwk_recovery_disable(BOOL_T en)
{
    g_disable_recovery = en;
}

BOOL_T tkl_zg_nwk_recovery_is_available(VOID_T)
{
    TUYA_RECOVERY_INFO_T nwk_info;
    
    if(g_disable_recovery) {
        return FALSE;
    }

    halCommonGetToken(&nwk_info, TOKEN_NETWORK_RECOVERY_INFO);
    tkl_nwk_reco_debug("recovery node type: 0x%x\r\n", nwk_info.node_type);
    if(nwk_info.node_type == 0 || nwk_info.node_type == 0xFF) {
        return FALSE;
    }
    return TRUE;
}

VOID_T tkl_zg_nwk_recovery_start(VOID_T)
{
    BOOL_T status;
    EmberNetworkParameters nwk_params = {0};

    tokTypeStackNodeData node_info;
    tokTypeStackTrustCenter tc_info;
    TUYA_RECOVERY_INFO_T nwk_info;

    if(g_disable_recovery) {
        return;
    }

    halCommonGetToken(&nwk_info, TOKEN_NETWORK_RECOVERY_INFO);
    if(nwk_info.node_type == 0 || nwk_info.node_type == 0xFF) {
        return;
    }

    halCommonGetToken(&node_info, TOKEN_STACK_NODE_DATA);
    halCommonGetToken(&tc_info, TOKEN_STACK_TRUST_CENTER);

    if(nwk_info.node_type == EMBER_ROUTER) {
        nwk_params.joinMethod = EMBER_USE_CONFIGURED_NWK_STATE;
    }
    else if(nwk_info.node_type >= EMBER_END_DEVICE) {
        nwk_params.joinMethod = EMBER_USE_NWK_REJOIN_HAVE_NWK_KEY;
    }
    
    nwk_params.radioChannel = node_info.radioFreqChannel;
    nwk_params.radioTxPower = node_info.radioTxPower;
    nwk_params.panId = node_info.panId;
    nwk_params.nwkManagerId = 0x0000;
    nwk_params.nwkUpdateId = 0;
    tkl_system_memcpy(nwk_params.extendedPanId, node_info.extendedPanId, 8);

    status = __network_security_set(nwk_info.tcl_key, nwk_info.nwk_key, tc_info.eui64);
    if(status) {
        __network_info_recovery(&nwk_params, nwk_info.node_type, node_info.zigbeeNodeId);
    }
    if(tkl_zg_node_type_get() == ZG_ROUTER) {
        tkl_network_recovery_reset_event_handler();
    }
    else {
        emberEventControlSetDelayMS(tkl_network_recovery_reset_event_control, 1000);
    }
}

VOID_T tkl_zg_nwk_recovery_info_save(VOID_T)
{
    tokTypeStackKeys stack_key;
    tokTypeStackNodeData node_info;
    tokTypeStackTrustCenter tc_info;
    TUYA_RECOVERY_INFO_T nwk_info;
    
    if(g_disable_recovery) {
        return;
    }

    halCommonGetToken(&stack_key, TOKEN_STACK_KEYS);
    halCommonGetToken(&node_info, TOKEN_STACK_NODE_DATA);
    halCommonGetToken(&tc_info, TOKEN_STACK_TRUST_CENTER);

#if defined (TUYA_RUNTIME_DBG)
    uint8_t i;
    tkl_nwk_reco_debug("activeKeySeqNum: %d\r\n",stack_key.activeKeySeqNum);
    tkl_nwk_reco_debug("Nwk key: ");
    for(i=0; i<16; i++) {
        tkl_nwk_reco_debug("0x%x ",stack_key.networkKey[i]);
    }
    tkl_nwk_reco_debug("\r\n\r\n");

    tkl_nwk_reco_debug("Panid: 0x%02x\r\n",node_info.panId);
    tkl_nwk_reco_debug("Txpower: %d\r\n",node_info.radioTxPower);
    tkl_nwk_reco_debug("Channel: %d\r\n",node_info.radioFreqChannel);
    tkl_nwk_reco_debug("Node type: %d\r\n",node_info.nodeType);
    tkl_nwk_reco_debug("Node id: 0x%02x\r\n",node_info.zigbeeNodeId);
    tkl_nwk_reco_debug("Ext Panid: ");
    for(i=0; i<8; i++) {
        tkl_nwk_reco_debug("0x%x ",node_info.extendedPanId[i]);
    }
    tkl_nwk_reco_debug("\r\n\r\n");

    tkl_nwk_reco_debug("TC mode: %d\r\n",tc_info.mode);
    tkl_nwk_reco_debug("TC addr: ");
    for(i=0; i<8; i++) {
        tkl_nwk_reco_debug("0x%x ",tc_info.eui64[i]);
    }
    tkl_nwk_reco_debug("\r\n");
    tkl_nwk_reco_debug("TC key: ");
    for(i=0; i<16; i++) {
        tkl_nwk_reco_debug("0x%x ",tc_info.key[i]);
    }
    tkl_nwk_reco_debug("\r\n\r\n");
#endif
    nwk_info.node_type = node_info.nodeType;
    tkl_system_memcpy(nwk_info.tcl_key, tc_info.key, 16);
    tkl_system_memcpy(nwk_info.nwk_key, stack_key.networkKey, 16);
    tkl_nwk_reco_debug("recovery nwk info save: node type 0x%x\r\n",nwk_info.node_type);
    halCommonSetToken(TOKEN_NETWORK_RECOVERY_INFO, &nwk_info);
}

VOID_T tkl_zg_nwk_recovery_info_clear(VOID_T)
{
    TUYA_RECOVERY_INFO_T nwk_info;
    
    if(g_disable_recovery) {
        return;
    }
    tkl_nwk_reco_debug("recovery nwk info clear.\r\n");
    tkl_system_memset(&nwk_info, 0, sizeof(TUYA_RECOVERY_INFO_T));
    halCommonSetToken(TOKEN_NETWORK_RECOVERY_INFO, &nwk_info);
}

BOOL_T tkl_zg_nwk_out_of_band_join(TKL_OUTOF_BAND_JOIN_T *info)
{
    EmberNetworkParameters nwk_params = {0};
    
    // Global TCLink key
    UINT8_T tcl_key[16] = {0x5A, 0x69, 0x67, 0x42, 0x65, 0x65, 0x41, 0x6C,
                          0x6C, 0x69, 0x61, 0x6E, 0x63, 0x65, 0x30, 0x39};

    if(info->node_type == ZG_ROUTER) {
        nwk_params.joinMethod = EMBER_USE_CONFIGURED_NWK_STATE;
    }
    else if(info->node_type == ZG_SLEEPY_END_DEVICE) {
        nwk_params.joinMethod = EMBER_USE_NWK_REJOIN_HAVE_NWK_KEY;
    }
    nwk_params.radioChannel = info->channel;
    nwk_params.radioTxPower = info->tx_power;
    nwk_params.panId = info->panid;
    nwk_params.nwkManagerId = 0x0000;
    nwk_params.nwkUpdateId = 0;
    tkl_system_memcpy(nwk_params.extendedPanId, info->ext_panid, 8);

    if(__network_security_set(tcl_key, info->nwk_key, info->tc_mac)) {
        if(__network_info_recovery(&nwk_params, info->node_type+2, info->node_id)) {
            emberSendDeviceAnnouncement();
            return TRUE;
        }
    }

    return FALSE;
}

