/*
 * @Author: Deven
 * @Email: liming@tuya.com
 * @LastEditors: Please set LastEditors
 * @FileName: &file name&
 * @Description: 
 * @Copyright: HANGZHOU TUYA INFORMATION TECHNOLOGY CO.,LTD
 * @Company: http://www.tuya.com
 * @Date: 2021-04-19 16:12:17
 * @LastEditTime: 2022-01-13 10:20:58
 */
#include "tkl_platform_types.h"
#include "tkl_zigbee_inner.h"
#include "tkl_memory.h"
#include "tkl_zcl_identify.h"
#include "tkl_endpoint_register.h"

#include "tkl_zll_commissioning.h"

#if defined (TUYA_RUNTIME_DBG)
#include "tkl_uart.h"
#define tkl_zll_debug(...)      tkl_dbg_printf(__VA_ARGS__)
#else
#define tkl_zll_debug(...)
#endif

#define ZLL_MALLOC      tkl_system_malloc
#define ZLL_FREE        tkl_system_free

#define DEFAULT_IDENTIFY_DURATION_S 30

extern INT8_T emZllRssiThreshold;       // ZLL server rssi threshold (vendor sdk internal variable)
extern INT8_T emAfZllClientScanPower;   // ZLL client radio power (vendor sdk internal variable)

extern zll_dev_t g_zll_dev;
extern void emberAfPluginInterpanInit(void);
extern void emberAfPluginZllCommissioningCommonInit(void);

typedef enum {
    ZLL_AS_TARGET_ROLE = 0,
    ZLL_AS_INITIATOR_ROLE = 0x01
} __ZLL_ROLE_E;

typedef struct {
    UINT8_T is_init;
    UINT8_T is_zll_3_0_test;
    __ZLL_ROLE_E zll_role;
    TKL_ZLL_TARGET_CMP_CB target_cmp_cb;
    TKL_ZLL_INITIATOR_CMP_CB initiator_cmp_cb;
    TKL_ZLL_POWER_STRATETGY_CB power_strategy_cb;
} __ZLL_MGR_T;

__ZLL_MGR_T g_zll_mgr;

ZLL_EXT_CB_T g_zll_ext_cb;

SDK_COMMAND_CB_T g_zll_cb;

// TODO: vendor sdk have a bug: client scan primarychannel will be return error in :
//   status = emberSetLogicalAndRadioChannel(emAfZllNetwork.zigbeeNetwork.channel);
//   debugPrintln("set log channel status %d, channel %d\r\n",status,emAfZllNetwork.zigbeeNetwork.channel);



/** @brief Identify
 *
 * This function is called by the ZLL Commissining Server plugin to notify the
 * application that it should take an action to identify itself. This typically
 * occurs when an Identify Request is received via inter-PAN messaging.
 *
 * @param durationS If the duration is zero, the device should exit identify
 * mode. If the duration is 0xFFFF, the device should remain in identify mode
 * for the default time. Otherwise, the duration specifies the length of time
 * in seconds that the device should remain in identify mode. Ver.: always
 */
void emberAfPluginZllCommissioningServerIdentifyCallback(uint16_t durationS)
{
    UINT8_T i, ep_id;

    tkl_zll_debug("zll identify times %d\r\n", durationS);

    if(g_zll_ext_cb.identify_request_cb){
        g_zll_ext_cb.identify_request_cb(durationS);
    }

    for(i = 0; i < tkl_zg_endpoint_count_get(); i++) {
        ep_id = emberAfEndpointFromIndex(i);
        if (emberAfContainsServer(ep_id, CLUSTER_ZLL_COMMISSIONING_CLUSTER_ID)) {
            // if the value of this field is 0x0000, the device shall immediately stop its identify operation. 
            // if the value of this field is 0xffff, the device shall identify itself for an application specific time. 
            //for all other values, the device shall identify itself for a time equal to this value in units of 1 second.
            tkl_zg_identify_start(ep_id, (durationS == 0XFFFF) ? DEFAULT_IDENTIFY_DURATION_S: durationS);
        }
    }
}


/** @brief Reset To Factory New
 *
 * This function is called by the ZLL Commissioning Common plugin when a request to
 * reset to factory new is received. The plugin will leave the network, reset
 * attributes managed by the framework to their default values, and clear the
 * group and scene tables. The application should perform any other necessary
 * reset-related operations in this callback, including resetting any
 * externally-stored attributes.
 *
 */
void emberAfPluginZllCommissioningCommonResetToFactoryNewCallback(void)
{
    tkl_zll_debug("zll reset to factory new\r\n");
     if(g_zll_mgr.power_strategy_cb != NULL){
        g_zll_mgr.power_strategy_cb(FALSE);
    }
    if (g_zll_mgr.target_cmp_cb != NULL) {
        g_zll_mgr.target_cmp_cb(TKL_ZLL_TARGET_LEAVE);
    }
}

/** @brief Touch Link Complete
 *
 * This function is called by the ZLL Commissioning Common plugin when touch linking
 * completes.
 *
 * @param networkInfo The ZigBee and ZLL-specific information about the network
 * and target. Ver.: always
 * @param deviceInformationRecordCount The number of sub-device information
 * records for the target. Ver.: always
 * @param deviceInformationRecordList The list of sub-device information
 * records for the target. Ver.: always
 */
void emberAfPluginZllCommissioningCommonTouchLinkCompleteCallback(const EmberZllNetwork *networkInfo,
                                                                  uint8_t deviceInformationRecordCount,
                                                                  const EmberZllDeviceInfoRecord *deviceInformationRecordList)
{
    tkl_zll_debug("zll touchlink completed: chan = %d, node type = %d, state = %02X, devnum %d\r\n", networkInfo->zigbeeNetwork.channel, networkInfo->nodeType, networkInfo->state, deviceInformationRecordCount);

    tkl_zll_debug("zll role %d\r\n",g_zll_mgr.zll_role);
    if(g_zll_mgr.zll_role == ZLL_AS_INITIATOR_ROLE) {    // client 
        TKL_ZLL_DEV_LIST_T dev_list;
        TKL_ZLL_DEV_INFO_T *dev_info = NULL;
        
        if (g_zll_mgr.initiator_cmp_cb != NULL) {
            
            dev_list.dev_sums = deviceInformationRecordCount;
            dev_info = (TKL_ZLL_DEV_INFO_T *)ZLL_MALLOC(sizeof(TKL_ZLL_DEV_INFO_T) * dev_list.dev_sums);
            if(dev_info == NULL) {
                return;
            }

            for(UINT8_T i = 0; i < dev_list.dev_sums; i++) {
                tkl_system_memcpy(&dev_info[i].long_addr, &deviceInformationRecordList[i].ieeeAddress, 8);
                dev_info[i].ep_num = 1;
                dev_info[i].ep_list[0] = deviceInformationRecordList[i].endpointId;
                if (networkInfo->nodeType == EMBER_ROUTER) {
                    dev_info[i].type = ZG_ROUTER;
                }
                else if (networkInfo->nodeType == EMBER_END_DEVICE) {
                    dev_info[i].type = ZG_END_DEVICE;
                }
                else if (networkInfo->nodeType == EMBER_SLEEPY_END_DEVICE) {
                    dev_info[i].type = ZG_SLEEPY_END_DEVICE;
                }
                else {
                    dev_info[i].type = ZG_UNKNOWN_DEVICE;
                }
                dev_info[i].short_addr = networkInfo->nodeId;
                dev_info[i].channel = networkInfo->zigbeeNetwork.channel;
            }
             if(g_zll_mgr.power_strategy_cb != NULL){
                g_zll_mgr.power_strategy_cb(TRUE);
            }
            g_zll_mgr.initiator_cmp_cb(TKL_ZLL_SCAN_RESULT_SUCESS, &dev_list);
        }
        g_zll_mgr.zll_role = ZLL_AS_TARGET_ROLE;    // set default role
    }
    else {
         if(g_zll_mgr.power_strategy_cb != NULL){
            g_zll_mgr.power_strategy_cb(TRUE);
        }

        if (g_zll_mgr.target_cmp_cb != NULL) {
            g_zll_mgr.target_cmp_cb(TKL_ZLL_TARGET_JOINED);
        }
    }
}

/** @brief Touch Link Failed
 *
 * This function is called by the ZLL Commissioning Client plugin if touch linking
 * fails.
 *
 * @param status The reason the touch link failed. Ver.: always
 */
void emberAfPluginZllCommissioningClientTouchLinkFailedCallback(EmberAfZllCommissioningStatus status)
{
    tkl_zll_debug("zll touchlink failed %d\r\n",status);
    g_zll_mgr.zll_role = ZLL_AS_TARGET_ROLE;    // set default role
    if(g_zll_mgr.power_strategy_cb != NULL){
        g_zll_mgr.power_strategy_cb(FALSE);
    }
    if (g_zll_mgr.initiator_cmp_cb != NULL) {
        g_zll_mgr.initiator_cmp_cb(TKL_ZLL_SCAN_RESULT_NO_FOUND, NULL);
    }
}


/** @brief Initial Security State
 *
 * This function is called by the ZLL Commissioning Common plugin to determine the
 * initial security state to be used by the device. The application must
 * populate the ::EmberZllInitialSecurityState structure with a configuration
 * appropriate for the network being formed, joined, or started. Once the
 * device forms, joins, or starts a network, the same security configuration
 * will remain in place until the device leaves the network.
 *
 * @param securityState The security configuration to be populated by the
 * application and ultimately set in the stack. Ver.: always
 */
void emberAfPluginZllCommissioningCommonInitialSecurityStateCallback(EmberZllInitialSecurityState *securityState)
{
    // set zigbee 3.0 test encryption key 
    UINT8_T encryption_key[] = { 0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 
                                 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF };
    
    // set zigbee 3.0 test pre link key 
    UINT8_T pre_link_key[] = { 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7,    
                               0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF };
    if(g_zll_mgr.is_zll_3_0_test) {
        securityState->keyIndex = EMBER_ZLL_KEY_INDEX_CERTIFICATION;
        tkl_system_memcpy(securityState->encryptionKey.contents, encryption_key, 16);
        tkl_system_memcpy(securityState->preconfiguredKey.contents, pre_link_key, 16);
    }
}


STATIC VOID_T __zll_commissioning_init(VOID_T) 
{
    if(!g_zll_mgr.is_init) {
        g_zll_mgr.is_init = TRUE;
        emberAfPluginZllCommissioningCommonInit();
        emberAfPluginInterpanInit();
    }
}

BOOL_T tkl_zll_commissioning_rx_on_set(UINT32_T time_ms)
{
    return emberZllSetRxOnWhenIdle(time_ms) == EMBER_INVALID_CALL ?FALSE: TRUE;
}

OPERATE_RET tkl_zll_target_touchlink_init(VOID_T)
{
    OPERATE_RET ret = OPRT_OK;
    UINT8_T i, ep_id;

    tkl_zll_debug("zll target init.\r\n");
    __zll_commissioning_init();

    // NOTE: ZLL commissioning cluster must at same endpoint  
    for(i = 0; i < tkl_zg_endpoint_count_get(); i++) {
        ep_id = emberAfEndpointFromIndex(i);
        if(emberAfContainsServer(ep_id, (EmberAfClusterId)CLUSTER_ZLL_COMMISSIONING_CLUSTER_ID)) {
            tkl_zll_debug("zll server init.\r\n");
            ret = tkl_zg_identify_init();

            g_zll_dev.type_mask |= ZLL_SERVER_MASK;
            g_zll_cb.server_parse_cb = emberAfZllCommissioningClusterServerCommandParse;
        }
    }
    // even if there is no server cluster, it is also set to target by default, 
    // and the initiator will be reset to the initiator role when calling the API
    g_zll_mgr.zll_role = ZLL_AS_TARGET_ROLE;    // set default role
    
    return ret;
}

OPERATE_RET tkl_zll_initiator_touchlink_init(VOID_T)
{
    OPERATE_RET ret = OPRT_OK;
    UINT8_T i, ep_id;

    __zll_commissioning_init();
    tkl_zll_debug("zll initiator init.\r\n");
    
    // NOTE: ZLL commissioning cluster must at same endpoint  
    for(i = 0; i < tkl_zg_endpoint_count_get(); i++) {
        ep_id = emberAfEndpointFromIndex(i);
        if(emberAfContainsClient(ep_id, (EmberAfClusterId)CLUSTER_ZLL_COMMISSIONING_CLUSTER_ID)) {
            tkl_zll_debug("zll client init.\r\n");
            g_zll_dev.type_mask |= ZLL_CLIENT_MASK;
            g_zll_cb.client_parse_cb = emberAfZllCommissioningClusterClientCommandParse;
        }
    }
    
    return ret;
}


BOOL_T tkl_zll_commissioning_in_process(VOID_T)
{
    if (emberZllOperationInProgress() || 
        emberAfZllTouchLinkInProgress()) {
        return TRUE;
    }

    return FALSE;
}

VOID_T tkl_zll_commissioning_3_0_test_enable(VOID_T)
{
    g_zll_mgr.is_zll_3_0_test = TRUE;
}


// for zll touchlink server
BOOL_T tkl_zll_target_touchlink_enable(BOOL_T en)
{
    return emberAfZllDisable() ? FALSE: TRUE;
}
BOOL_T tkl_zll_target_touchlink_is_enable(VOID_T)
{
    return emberAfZllEnable() ? FALSE: TRUE;
}

INT8_T tkl_zll_target_rssi_threshold_get(VOID_T)
{
    return emZllRssiThreshold;
}
VOID_T tkl_zll_target_rssi_threshold_set(INT8_T rx_rssi)
{
    emZllRssiThreshold = rx_rssi;
}

BOOL_T tkl_zll_target_no_reset_for_non_factory_new(VOID_T)
{
    return emberAfZllNoTouchlinkForNFN() ? FALSE: TRUE;
}

BOOL_T tkl_zll_target_no_touchlink_for_non_factory_new(VOID_T)
{
    return emberAfZllNoResetForNFN() ? FALSE: TRUE;
}

OPERATE_RET tkl_zll_target_register_complete_cb(TKL_ZLL_TARGET_CMP_CB cb)
{
    if(cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_zll_mgr.target_cmp_cb = cb;

    return OPRT_OK;
}

OPERATE_RET tkl_zll_target_register_identify_req_cb(TKL_ZLL_IDENTIFY_REQ_CB cb)
{
    if(cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_zll_ext_cb.identify_request_cb = cb;

    return OPRT_OK;
}

OPERATE_RET tkl_zll_target_register_reset_to_fn_cb(TKL_ZLL_RESET_TO_FN_CB cb)
{
    if(cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_zll_ext_cb.reset_to_fn_cb = cb;

    return OPRT_OK;
}

// for zll touchlink client
INT8_T tkl_zll_initiator_tx_power_get(VOID_T)
{
    return emAfZllClientScanPower;
}

VOID_T tkl_zll_initiator_tx_power_set(INT8_T tx_db)
{
    emAfZllClientScanPower = tx_db;
}
VOID_T tkl_zll_initiator_scan_stop(VOID_T)
{
    emberAfZllAbortTouchLink();
}

TKL_ZLL_SCAN_STATUS_E tkl_zll_initiator_scan_start(TKL_ZLL_SCAN_TYPE_E type)
{
    EmberStatus ret;

    g_zll_mgr.zll_role = ZLL_AS_INITIATOR_ROLE;

    if (type == TKL_ZLL_SCANE_FOR_JOIN) {
        ret = emberAfZllInitiateTouchLink();
    }
    else if (type == TKL_ZLL_SCAN_FOR_LEAVE) {
        ret = emberAfZllResetToFactoryNewRequest();
    }
    else if (type == TKL_ZLL_SCAN_FOR_INDENTIFY) {
        ret = emberAfZllIdentifyRequest();
    }

    return (ret == EMBER_SUCCESS) ? TKL_ZLL_SCAN_OK : TKL_ZLL_SCAN_BUSY;
}

OPERATE_RET tkl_zll_initiator_register_complete_cb(TKL_ZLL_INITIATOR_CMP_CB cb)
{
    if(cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_zll_mgr.initiator_cmp_cb = cb;

    return OPRT_OK;
}
OPERATE_RET tkl_zll_register_power_stratege_cb(TKL_ZLL_POWER_STRATETGY_CB cb)
{
    if(cb == NULL) {
        return OPRT_INVALID_PARM;
    }
    g_zll_mgr.power_strategy_cb = cb;

    return OPRT_OK;
}

