/***********************************************************
*  File: tuya_uart.c
*  Author: nzy
*  Date: 20171106
***********************************************************/
#define __TUYA_UART_GLOBALS
#include "tuya_uart.h"
#include "tuya_hal_uart.h"
#include "uni_log.h"
#include "tuya_hal_semaphore.h"
#include "mem_pool.h"


/***********************************************************
*************************micro define***********************
***********************************************************/
typedef struct {
    UINT_T buf_len;
    BYTE_T *buf;
    USHORT_T in;
    USHORT_T out;
    BOOL_T RUnBlock;
    SEM_HANDLE uart_sem;
    BOOL_T has_sem_get;
}TUYA_UART_S;

/***********************************************************
*************************variable define********************
***********************************************************/
STATIC TUYA_UART_S ty_uart[TY_UART_NUM];

/***********************************************************
*************************function define********************
***********************************************************/
STATIC VOID __ty_uart_irq(TY_UART_PORT_E port, UCHAR_T ch);


/***********************************************************
*  Function: ty_uart_init 
*  Input: port badu bits parity stop
*  Output: none
*  Return: OPERATE_RET
***********************************************************/
OPERATE_RET ty_uart_init(TY_UART_PORT_E port,TY_UART_CFG_S *cfg, UINT_T bufsz, BOOL_T RUnBlock)
{
    PR_DEBUG("ty_uart_init, cfg:0x%x, port:%d, bufsz:%d", cfg, port, bufsz);
    if((cfg == NULL) || (port >= TY_UART_NUM) || (bufsz == 0)) {
        return OPRT_INVALID_PARM;
    }
    
    if(ty_uart[port].buf == NULL) {
        memset(&ty_uart[port], 0, sizeof(TUYA_UART_S));
        ty_uart[port].buf = Malloc(bufsz);
        if(ty_uart[port].buf == NULL) {
            return OPRT_MALLOC_FAILED;
        }
        ty_uart[port].buf_len = bufsz;
    }else {
        return OPRT_COM_ERROR;
    }
   
    if(!RUnBlock) {
        OPERATE_RET op_ret = tuya_hal_semaphore_create_init(&(ty_uart[port].uart_sem), 0, 10);
        if(OPRT_OK != op_ret) {
            PR_ERR("tuya_hal_semaphore_create_init failed %d",op_ret);
            Free(ty_uart[port].buf);
            ty_uart[port].buf = NULL;
            return op_ret;
        }
    }
    
    ty_uart[port].in = 0;
    ty_uart[port].out = 0;
    ty_uart[port].RUnBlock = RUnBlock;
    ty_uart[port].has_sem_get = TRUE;
    
    TY_UART_DEV_S uart_s = {0};
    memcpy(&(uart_s.cfg), cfg, SIZEOF(TY_UART_CFG_S));
    uart_s.port = port;
    uart_s.rx_cb = __ty_uart_irq;
    
    INT_T ret = tuya_hal_uart_init(&uart_s);
    if(ret != 0) {
        PR_ERR("tuya_hal_uart_init failed %d",ret);
        Free(ty_uart[port].buf);
        ty_uart[port].buf = NULL;
        if(!ty_uart[port].RUnBlock) {
            tuya_hal_semaphore_release(ty_uart[port].uart_sem);
        }
        return ret;
    }

    return OPRT_OK;
}

/***********************************************************
*  Function: ty_uart_free 
*  Input:free uart
*  Output: none
*  Return: OPERATE_RET
***********************************************************/
OPERATE_RET ty_uart_free(IN CONST TY_UART_PORT_E port)
{
    if(port >= TY_UART_NUM) {
       return OPRT_INVALID_PARM;
    }
    
    INT_T ret = tuya_hal_uart_deinit(port);
    if(ret != 0) {
         PR_ERR("tuya_hal_uart_deinit failed %d",ret);
         return ret;
    }
    
    if(ty_uart[port].buf != NULL) {
        Free(ty_uart[port].buf);
        ty_uart[port].buf = NULL;
    }
    ty_uart[port].buf_len = 0;
    
    if(!ty_uart[port].RUnBlock) {
        tuya_hal_semaphore_release(ty_uart[port].uart_sem);
    }

   return OPRT_OK;
}

/***********************************************************
*  Function: ty_uart_send_data 
*  Input: port data len
*  Output: none
*  Return: none
***********************************************************/
VOID ty_uart_send_data(IN CONST TY_UART_PORT_E port,IN CONST BYTE_T *data,IN CONST UINT_T len)
{
    if(port >= TY_UART_NUM) {
        return;
    }

    UINT_T i = 0;
    for(i = 0;i < len;i++) {
       tuya_hal_uart_send_char(port, *(data+i));
    }
}

STATIC UINT_T __ty_uart_read_data_size(IN CONST TY_UART_PORT_E port)
{
    UINT_T remain_buf_size = 0;

    if(ty_uart[port].in >= ty_uart[port].out) {
        remain_buf_size = ty_uart[port].in-ty_uart[port].out;
    }else {
        remain_buf_size = ty_uart[port].in + ty_uart[port].buf_len - ty_uart[port].out;
    }

    return remain_buf_size;
}

STATIC VOID __ty_uart_irq(TY_UART_PORT_E port, UCHAR_T ch)
{
    UCHAR_T rc = ch;
    TY_UART_PORT_E i = port;
    
    if(__ty_uart_read_data_size(i) < ty_uart[i].buf_len - 1) {
        ty_uart[i].buf[ty_uart[i].in++] = rc;
        if(ty_uart[i].in >= ty_uart[i].buf_len) {
            ty_uart[i].in = 0;
        }
        
        if(!ty_uart[i].RUnBlock) {
            if(ty_uart[i].has_sem_get) {
                ty_uart[i].has_sem_get = FALSE;
                tuya_hal_semaphore_post(ty_uart[i].uart_sem);
            }
        }
        
    }
    
}


/***********************************************************
*  Function: ty_uart_send_data 
*  Input: len->data buf len
*  Output: buf->read data buf
*  Return: actual read data size
***********************************************************/
UINT_T ty_uart_read_data(IN CONST TY_UART_PORT_E port,OUT BYTE_T *buf,IN CONST UINT_T len)
{
    if(NULL == buf || 0 == len) {
        return 0;
    }
    
    if(!ty_uart[port].RUnBlock) {
        OPERATE_RET op_ret = tuya_hal_semaphore_wait(ty_uart[port].uart_sem);
        if(OPRT_OK != op_ret) {
            PR_ERR("tuya_hal_semaphore_wait failed %d",op_ret);
            return op_ret;
        }
        
        ty_uart[port].has_sem_get = TRUE;
    }

    UINT_T actual_size = 0;
    UINT_T cur_num = __ty_uart_read_data_size(port);

    if(len > cur_num) {
        actual_size = cur_num;
    }else {
        actual_size = len;
    }
    //PR_NOTICE("uart_num = %d", cur_num);
    UINT_T i = 0;
    for(i = 0;i < actual_size;i++) {
        *buf++ = ty_uart[port].buf[ty_uart[port].out++];
        if(ty_uart[port].out >= ty_uart[port].buf_len) {
            ty_uart[port].out = 0;
        }
    }

    return actual_size;
}



