/**
 * @file tuya_hal_gpio.h
 * @brief GPIO操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */
#ifndef TUYA_HAL_GPIO_H
#define TUYA_HAL_GPIO_H
#include <stdint.h>
#include <stdbool.h>



#ifdef __cplusplus
    extern "C" {
#endif


/***********************************************************
*************************micro define***********************
***********************************************************/
typedef enum {
    TY_GPIOA_0 = 0,
    TY_GPIOA_1,
    TY_GPIOA_2,
    TY_GPIOA_3,
    TY_GPIOA_4,
    TY_GPIOA_5,
    TY_GPIOA_6,
    TY_GPIOA_7,
    TY_GPIOA_8,
    TY_GPIOA_9,
    TY_GPIOA_10,
    TY_GPIOA_11,
    TY_GPIOA_12,
    TY_GPIOA_13,
    TY_GPIOA_14,
    TY_GPIOA_15,
    TY_GPIOA_16,
    TY_GPIOA_17,
    TY_GPIOA_18,
    TY_GPIOA_19,
    TY_GPIOA_20,
    TY_GPIOA_21,
    TY_GPIOA_22,
    TY_GPIOA_23,
    TY_GPIOA_24,
    TY_GPIOA_25,
    TY_GPIOA_26,
    TY_GPIOA_27,
    TY_GPIOA_28,
    TY_GPIOA_29,
    TY_GPIOA_30,
    TY_GPIOA_31,
    TY_GPIOB_0 = 32,
    TY_GPIOB_1,
    TY_GPIOB_2,
    TY_GPIOB_3,
    TY_GPIOB_4,
    TY_GPIOB_5,
    TY_GPIOB_6,
    TY_GPIOB_7,
    TY_GPIOB_8,
}TY_GPIO_PORT_E;

typedef enum {
    TY_GPIO_PULLUP,                    /*!< Pad pull up            */
    TY_GPIO_PULLDOWN,                  /*!< Pad pull down          */
    TY_GPIO_PULLUP_PULLDOWN,           /*!< Pad pull up + pull down*/
    TY_GPIO_OPENDRAIN,                 /*!< Pad open drain         */
    TY_GPIO_FLOATING,                  /*!< Pad floating           */
} TY_GPIO_MODE_E;

typedef enum {
    IRQ_RISE_EVENT,
    IRQ_FALL_EVENT,
    IRQ_HIGH_EVENT,
    IRQ_LOW_EVENT
} TY_GPIO_IRQ_EVENT;

typedef enum {
    TY_IRQ_NONE,
    /** IRQ occurs on rising edge */
    TY_IRQ_RISE,
    /** IRQ occurs on falling edge */
    TY_IRQ_FALL,
    /** IRQ occurs on either edge */
    TY_IRQ_BOTH,
    /** IRQ occurs when line is low */
    TY_IRQ_HIGH,
    /** IRQ occurs when line is high */
    TY_IRQ_LOW
} TY_GPIO_IRQ_TRIG_TYPE;

/***********************************************************
*************************variable define********************
***********************************************************/

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief 用于设置gpio的输入输出
 * 
 * @param[in]       port     端口号
   @param[in]       in       输入/输出
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_inout_set(const TY_GPIO_PORT_E port, const bool in);

/**
 * @brief 用于设置gpio的上下拉模式
 * 
 * @param[in]       port     端口号
   @param[in]       mode     模式
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_mode_set(const TY_GPIO_PORT_E port, const TY_GPIO_MODE_E mode);

/**
 * @brief 用于读取gpio的电平
 * 
 * @param[in]       port     端口号
 * @return gpio的电平值 0/1
 */
int tuya_hal_gpio_read(const TY_GPIO_PORT_E port);

/**
 * @brief 用于设置gpio的电平
 * 
 * @param[in]       port     端口号
 * @param[in]       high     电平值
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_write(const TY_GPIO_PORT_E port, const bool high);

/**
 * @brief gpio中断回调
 * 
 * @param[in]       port     端口号
 * @param[in]       event    产生中断的原因
 */
typedef void (*TY_GPIO_CALLBACK)(TY_GPIO_PORT_E port, TY_GPIO_IRQ_EVENT event);

/**
 * @brief 用于初始化gpio的中断模式
 * 
 * @param[in]       port     端口号
 * @param[in]       cb       中断回调
 * @param[in]       trig_type     中断触发条件
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_irq_init(const TY_GPIO_PORT_E port, const TY_GPIO_CALLBACK cb, const TY_GPIO_IRQ_TRIG_TYPE trig_type);

#ifdef __cplusplus
}
#endif

#endif/*TUYA_HAL_GPIO_H*/

