/**
 * @file tuya_hal_uart.h
 * @brief UART操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#ifndef TUYA_HAL_UART_H
#define TUYA_HAL_UART_H

/***********************************************************
*************************micro define***********************
***********************************************************/
//TUYA default TY_UART0 is the log port, TY_UART1 is the account
//TY_UART2, TY_UART3 is the extension of the serial port
typedef enum {
    TY_UART0 = 0x00,
    TY_UART1,
    TY_UART2,
    TY_UART3,
    TY_UART_NUM,
} TY_UART_PORT_E;

#define TY_USER_UART TY_UART0

typedef enum {
    TY_UART_BAUDRATE_300        = 300,
    TY_UART_BAUDRATE_600        = 600,
    TY_UART_BAUDRATE_1200       = 1200,
    TY_UART_BAUDRATE_2400       = 2400,
    TY_UART_BAUDRATE_4800       = 4800,
    TY_UART_BAUDRATE_9600       = 9600,
    TY_UART_BAUDRATE_19200      = 19200,
    TY_UART_BAUDRATE_38400      = 38400,
    TY_UART_BAUDRATE_57600      = 57600,
    TY_UART_BAUDRATE_74880      = 74880,
    TY_UART_BAUDRATE_115200     = 115200,
    TY_UART_BAUDRATE_230400     = 230400,
    TY_UART_BAUDRATE_460800     = 460800,
    TY_UART_BAUDRATE_921600     = 921600,
    TY_UART_BAUDRATE_1500000    = 1500000,
    TY_UART_BAUDRATE_1843200    = 1843200,
    TY_UART_BAUDRATE_3686400    = 3686400,
} TY_UART_BAUDRATE_E;

typedef enum {
    TY_UART_DATA_BIT5           = 0x00,
    TY_UART_DATA_BIT6           = 0x01,
    TY_UART_DATA_BIT7           = 0x02,
    TY_UART_DATA_BIT8           = 0x03,
} TY_UART_DATA_BITS_E;

typedef enum {
    TY_UART_STOP_BIT1           = 0x01,
    TY_UART_STOP_BIT1_5         = 0x02,
    TY_UART_STOP_BIT2           = 0x03,
} TY_UART_STOP_BITS_E;

typedef enum {
    TY_UART_PARITY_NONE         = 0,
    TY_UART_PARITY_ODD          = 1,
    TY_UART_PARITY_EVEN         = 2,
} TY_UART_PARITY_E; 

typedef enum {
    TY_UART_FLOW_CONTROL_NONE    = 0,
    TY_UART_FLOW_CONTROL_ENABLE  = 1,
} TY_UART_FLOW_CONTROL_E;

typedef enum {
    TY_UART_INT_RX_DISABLE       = 0,
    TY_UART_INT_RX_ENABLE        = 1,
} TY_UART_INT_RX_E;


typedef struct {
    unsigned int              baudrate;
    unsigned short            data_bits   :4;
    unsigned short            stop_bits   :2;
    unsigned short            parity      :2;
    unsigned short            flowctl     :1;
    unsigned short            intrx       :1;
    unsigned short            reserved    :6;
} TY_UART_CFG_S;


typedef void (*UART_INT_RX_CB)(TY_UART_PORT_E port, unsigned char ch);

typedef struct TY_UART_DEV {
    TY_UART_PORT_E      port;
    TY_UART_CFG_S       cfg;
    UART_INT_RX_CB      rx_cb;
} TY_UART_DEV_S;
/***********************************************************
*************************variable define********************
***********************************************************/


/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief 用于初始化串口
 * 
 * @param[in]  uart     串口配置信息
 */
int tuya_hal_uart_init(const TY_UART_DEV_S *uart);


/**
 * @brief 用于释放串口
 * 
 * @param[in]  port     串口id
 */
int tuya_hal_uart_deinit(const TY_UART_PORT_E port);

/**
 * @brief 用于发送一个字节
 * 
 * @param[in]  port     串口id
 * @param[ch]  ch       要发送的数据
 */
int tuya_hal_uart_send_char(const TY_UART_PORT_E port, const unsigned char ch);

/**
 * @brief 用于轮询读一个字节
 * 
 * @param[in]   port     串口id
 * @param[out]  ch       读到的数据
 */
int tuya_hal_uart_pool_read_char(const TY_UART_PORT_E port, unsigned char *ch);

#endif/*TUYA_HAL_UART_H*/

