/**
 * @file tuya_hal_network.h
 * @brief 网络操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */


#ifndef __TUYA_HAL_NETWORK_H__
#define __TUYA_HAL_NETWORK_H__


#include <stdint.h>
#include <stdbool.h>


#ifdef __cplusplus
    extern "C" {
#endif


/* tuya sdk definition of 127.0.0.1 */
#define TY_IPADDR_LOOPBACK     ((unsigned int)0x7f000001UL)
/* tuya sdk definition of 0.0.0.0 */
#define TY_IPADDR_ANY          ((unsigned int)0x00000000UL)
/* tuya sdk definition of 255.255.255.255 */
#define TY_IPADDR_BROADCAST    ((unsigned int)0xffffffffUL)

//the MAC addr len
#define TY_MAC_ADDR_LEN 6

/* tuya sdk definition of socket protocol */
typedef enum {
    PROTOCOL_TCP = 0,
    PROTOCOL_UDP = 1,
}UNW_PROTOCOL_TYPE;

/* tuya sdk definition of transfer type */
typedef enum {
    TRANS_RECV = 0,
    TRANS_SEND = 1,
}UNW_TRANS_TYPE_E;

/* tuya sdk definition of shutdown type */
#define UNW_SHUT_RD   0
#define UNW_SHUT_WR   1
#define UNW_SHUT_RDWR 2

/* tuya sdk definition of socket errno */
typedef int UNW_ERRNO_T;
#define UNW_SUCCESS       0
#define UNW_FAIL          -1
#define UNW_EINTR         -2
#define UNW_EBADF         -3
#define UNW_EAGAIN        -4
#define UNW_EFAULT        -5
#define UNW_EBUSY         -6
#define UNW_EINVAL        -7
#define UNW_ENFILE        -8
#define UNW_EMFILE        -9
#define UNW_ENOSPC        -10
#define UNW_EPIPE         -11
#define UNW_EWOULDBLOCK   -12
#define UNW_ENOTSOCK      -13
#define UNW_ENOPROTOOPT   -14
#define UNW_EADDRINUSE    -15
#define UNW_EADDRNOTAVAIL -16
#define UNW_ENETDOWN      -17
#define UNW_ENETUNREACH   -18
#define UNW_ENETRESET     -19
#define UNW_ECONNRESET    -20
#define UNW_ENOBUFS       -21
#define UNW_EISCONN       -22
#define UNW_ENOTCONN      -23
#define UNW_ETIMEDOUT     -24
#define UNW_ECONNREFUSED  -25
#define UNW_EHOSTDOWN     -26
#define UNW_EHOSTUNREACH  -27
#define UNW_ENOMEM        -28
#define UNW_EMSGSIZE      -29

/* fd 最大个数, 不同平台根据实际情况定义 */
#define UNW_FD_MAX_COUNT    (64)

/* tuya sdk definition of fd operations */
typedef struct {
    unsigned char placeholder[(UNW_FD_MAX_COUNT+7)/8];
} UNW_FD_SET_T;

#define UNW_FD_SET(n,p)     tuya_hal_net_fd_set(n, p)
#define UNW_FD_CLR(n, p)    tuya_hal_net_fd_clear(n, p)
#define UNW_FD_ISSET(n,p)   tuya_hal_net_fd_isset(n,p)
#define UNW_FD_ZERO(p)      tuya_hal_net_fd_zero(p)

/* tuya sdk definition of IP info */
typedef struct
{
    char ip[16];    /* ip addr:  xxx.xxx.xxx.xxx  */
    char mask[16];  /* net mask: xxx.xxx.xxx.xxx  */
    char gw[16];    /* gateway:  xxx.xxx.xxx.xxx  */
}NW_IP_S;

/* tuya sdk definition of MAC info */
typedef struct
{
    unsigned char mac[TY_MAC_ADDR_LEN]; /* mac address */
}NW_MAC_S;

/* tuya sdk definition of IP addr */
typedef unsigned int UNW_IP_ADDR_T;


/**
 * @brief 用于获取错误序号
 * 
 * @retval         errno
 */
UNW_ERRNO_T tuya_hal_net_get_errno(void);

/**
 * @brief : 用于ip字符串数据转换网络序ip地址(4B)
 * @param[in]      ip    ip字符串    "192.168.1.1"
 * @return  ip地址(4B)
 */
UNW_IP_ADDR_T tuya_hal_net_addr(const char *ip);

/**
 * @brief : Ascii网络字符串地址转换为主机序(4B)地址 
 * @param[in]            ip_str
 * @return   主机序ip地址(4B)
*/
UNW_IP_ADDR_T tuya_hal_net_str2addr(const char *ip);

/**
 * @brief : set fds
 * @param[in]      fd
 * @param[inout]      fds
 * @return  0: success  <0: fail
 */
int tuya_hal_net_fd_set(const int fd, UNW_FD_SET_T* fds);

/**
 * @brief : clear fds
 * @param[in]      fd
 * @param[inout]      fds
 * @return  0: success  <0: fail
 */
int tuya_hal_net_fd_clear(const int fd, UNW_FD_SET_T* fds);

/**
 * @brief : 判断fds是否被置位
 * @param[in]      fd
 * @param[in]      fds
 * @return  0-没有可读fd other-有可读fd
 */
int tuya_hal_net_fd_isset(const int fd, UNW_FD_SET_T* fds);

/**
 * @brief : init fds
 * @param[inout]      fds
 * @return  0: success  <0: fail
 */
int tuya_hal_net_fd_zero(UNW_FD_SET_T* fds);

/**
 * @brief : select
 * @param[in]         maxfd
 * @param[inout]      readfds
 * @param[inout]      writefds
 * @param[inout]      errorfds
 * @param[inout]      ms_timeout
 * @return  0: success  <0: fail
 */
int tuya_hal_net_select(const int maxfd, UNW_FD_SET_T *readfds, UNW_FD_SET_T *writefds,\
                        UNW_FD_SET_T *errorfds, const unsigned int ms_timeout);

/**
 * @brief : close fd
 * @param[in]      fd
 * @return  0: success  <0: fail
*/
int tuya_hal_net_close(const int fd);

/**
 * @brief : shutdow fd
 * @param[in]      fd
 * @param[in]      how
 * @return  OPRT_OK: success  <0: fail
*/
int tuya_hal_net_shutdown(const int fd, const int how);

/**
 * @brief : creat fd
 * @param[in]      type
 * @return  >=0: socketfd  <0: fail
*/
int tuya_hal_net_socket_create(const UNW_PROTOCOL_TYPE type);

/**
 * @brief : connect
 * @param[in]      fd
 * @param[in]      addr
 * @param[in]      port
 * @return  0: success  Other: fail
*/
int tuya_hal_net_connect(const int fd, const UNW_IP_ADDR_T addr, const unsigned short port);

/**
 * @brief : raw connect
 * @param[in]      fd
 * @param[in]      p_socket
 * @param[in]      len
 * @return  0: success  Other: fail
*/
int tuya_hal_net_connect_raw(const int fd, void *p_socket_addr, const int len);

/**
 * @brief : bind
 * @param[in]      fd
 * @param[in]      addr
 * @param[in]      port
 * @return  0: success  Other: fail
*/
int tuya_hal_net_bind(const int fd, const UNW_IP_ADDR_T addr, const unsigned short port);

/**
 * @brief : bind ip
 * @param[in]            fd
 * @param[inout]         addr
 * @param[inout]         port
 * @return  0: success  <0: fail
*/
int tuya_hal_net_socket_bind(const int fd, const char *ip);

/**
 * @brief : listen
 * @param[in]      fd
 * @param[in]      backlog
 * @return  0: success  < 0: fail
*/
int tuya_hal_net_listen(const int fd, const int backlog);

/**
 * @brief : accept
 * @param[in]            fd
 * @param[inout]         addr
 * @param[inout]         port
 * @return  >=0: 新接收到的socketfd  others: fail
*/
int tuya_hal_net_accept(const int fd, UNW_IP_ADDR_T *addr, unsigned short *port);

/**
 * @brief : send
 * @param[in]      fd
 * @param[in]      buf
 * @param[in]      nbytes
 * @return  nbytes has sended
*/
int tuya_hal_net_send(const int fd, const void *buf, const unsigned int nbytes);

/**
 * @brief : send to
 * @param[in]      fd
 * @param[in]      buf
 * @param[in]      nbytes
 * @param[in]      addr
 * @param[in]      port
 * @return  nbytes has sended
*/
int tuya_hal_net_send_to(const int fd, const void *buf, const unsigned int nbytes,\
                         const UNW_IP_ADDR_T addr,const unsigned short port);

/**
 * @brief : recv
 * @param[in]         fd
 * @param[inout]      buf
 * @param[in]         nbytes
 * @return  nbytes has received
*/
int tuya_hal_net_recv(const int fd, void *buf, const unsigned int nbytes);

/**
 * @brief : Receive enough data to specify
 * @param[in]            fd
 * @param[inout]         addr
 * @param[inout]         port
 * @return  nbytes has received
*/
int tuya_hal_net_recv_nd_size(const int fd, \
                              void *buf, \
                              const unsigned int buf_size, \
                              const unsigned int nd_size);

/**
 * @brief : recvfrom
 * @param[in]         fd
 * @param[inout]      buf
 * @param[in]         nbytes
 * @param[inout]         addr
 * @param[inout]         port
 * @return  nbytes has received
*/
int tuya_hal_net_recvfrom(const int fd, \
                          void *buf, \
                          const unsigned int nbytes,\
                          UNW_IP_ADDR_T *addr, \
                          unsigned short *port);

/**
 * @brief : get fd block info
 * @param[in]      fd
 * @return  <0-失败   >0-非阻塞    0-阻塞
*/
int tuya_hal_net_get_nonblock(const int fd);

/**
 * @brief : set block block or not
 * @param[in]      fd
 * @param[in]      block
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_block(const int fd, const bool block);

/**
 * @brief : set timeout
 * @param[in]         fd
 * @param[in]         ms_timeout
 * @param[in]         type
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_timeout(const int fd, \
                           const int ms_timeout,\
                           const UNW_TRANS_TYPE_E type);

/**
 * @brief : set buf size
 * @param[in]         fd
 * @param[in]         buf_size
 * @param[in]         type
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_bufsize(const int fd, \
                            const int buf_size,\
                            const UNW_TRANS_TYPE_E type);

/**
 * @brief : set reuse
 * @param[in]         fd
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_reuse(const int fd);

/**
 * @brief : disable nagle
 * @param[in]         fd
 * @return  0: success  <0: fail
*/
int tuya_hal_net_disable_nagle(const int fd);

/**
 * @brief : set broadcast
 * @param[in]         fd
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_boardcast(const int fd);


/**
 * @brief : tcp保活设置 
 * @param[in]            fd-the socket fd
 * @param[in]            alive-open(1) or close(0) 
 * @param[in]            idle-how long to send a alive packet(in seconds) 
 * @param[in]            intr-time between send alive packets (in seconds)
 * @param[in]            cnt-keep alive packets fail times to close the connection
 * @return  0: success  <0: fail
*/
int tuya_hal_net_set_keepalive(const int fd, \
                               const bool alive,\
                               const unsigned int idle, \
                               const unsigned int intr,\
                               const unsigned int cnt);

/**
 * @brief : dns parse
 * @param[in]            domain
 * @param[inout]         addr
 * @return  0: success  <0: fail
*/
int tuya_hal_net_gethostbyname(const char *domain, UNW_IP_ADDR_T *addr);

#ifdef __cplusplus
}
#endif

#endif // __TUYA_HAL_NETWORK_H__

