/**
 * @file tuya_hal_system.h
 * @brief 系统接口封装
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#ifndef _TUYA_HAL_SYSTEM_H
#define _TUYA_HAL_SYSTEM_H

#include <stdbool.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#if OPERATING_SYSTEM == SYSTEM_LINUX
typedef uint64_t SYS_TICK_T;
#else
typedef uint32_t SYS_TICK_T;
#endif

typedef enum {
    TY_RST_POWER_OFF = 0,
    TY_RST_HARDWARE_WATCHDOG,
    TY_RST_FATAL_EXCEPTION,
    TY_RST_SOFTWARE_WATCHDOG,
    TY_RST_SOFTWARE,
    TY_RST_DEEPSLEEP,
    TY_RST_HARDWARE,
    TY_RST_OTHER = 0xAA,
    TY_RST_UNSUPPORT = 0xFF,
} TY_RST_REASON_E;


typedef enum {
    TY_CPU_SLEEP,
    TY_CPU_DEEP_SLEEP,
} TY_CPU_SLEEP_MODE_E;

/**
 * @brief tuya_hal_get_systemtickcount用于获取系统运行ticket 
 * count
 * @return SYS_TICK_T 
 */
SYS_TICK_T tuya_hal_get_systemtickcount(void);

/**
 * @brief tuya_hal_get_tickratems用于获取系统ticket是多少个ms
 * 
 * @return  the time is ms of a system ticket
 */
unsigned int tuya_hal_get_tickratems(void);

/**
 * @brief tuya_hal_system_sleep用于系统sleep
 * 
 * @param[in] msTime sleep time is ms
 */
void tuya_hal_system_sleep(const unsigned long msTime);

/**
 * @brief tuya_hal_system_isrstatus用于检查系统是否处于中断上下文
 * 
 * @return true 
 * @return false 
 */
bool tuya_hal_system_isrstatus(void);

/**
 * @brief tuya_hal_system_reset用于重启系统
 * 
 */
void tuya_hal_system_reset(void);

/**
 * @brief tuya_hal_system_getheapsize用于获取堆大小/剩余内存大小
 * 
 * @return int <0: don't support  >=0: current heap size/free memory
 */
int tuya_hal_system_getheapsize(void);

/**
 * @brief tuya_hal_system_get_rst_info用于获取硬件重启原因
 * 
 * @return 硬件重启原因
 */
TY_RST_REASON_E tuya_hal_system_get_rst_info(void);

/**
 * @brief tuya_hal_get_random_data用于获取指定条件下的随机数
 * 
 * @param[in] range 
 * @return 随机值
 */
int tuya_hal_get_random_data(const unsigned int range);


/**
 * @brief tuya_hal_set_cpu_lp_mode用于设置cpu的低功耗模式
 * 
 * @param[in] en 
 * @param[in] mode
 * @return int 0=成功，非0=失败
 */
int tuya_hal_set_cpu_lp_mode(const bool en, const TY_CPU_SLEEP_MODE_E mode);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif

