/**
 * @file tuya_hal_gpio.c
 * @brief GPIO操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */
#define __TUYA_GPIO_GLOBALS
#include "gpio_api.h"
#include "tuya_os_adapter.h"
#include "gpio_irq_api.h"
#include "gpio_irq_ex_api.h"

#include "tuya_hal_gpio.h"

#include "tuya_os_adapter_errcode.h"

/***********************************************************
*************************micro define***********************
***********************************************************/
typedef struct {
    PinName pf_port; // platform port
    gpio_t *pf_obj;
    TY_GPIO_CALLBACK cb;
}TY_GPIO_MAP_S;


#define CNTSOF(a)   (sizeof(a)/sizeof(a[0]))
/***********************************************************
*************************variable define********************
***********************************************************/
TY_GPIO_MAP_S gpio_map_list[] = {
    {PA_0,NULL,NULL},
    {PA_1,NULL,NULL}, 
    {PA_2,NULL,NULL}, 
    {PA_3,NULL,NULL}, 
    {PA_4,NULL,NULL}, 
    {PA_5,NULL,NULL}, 
    {PA_6,NULL,NULL}, 
    {PA_7,NULL,NULL}, 
    {PA_8,NULL,NULL}, 
    {PA_9,NULL,NULL}, 
    {PA_10,NULL,NULL},
    {PA_11,NULL,NULL},
    {PA_12,NULL,NULL},
    {PA_13,NULL,NULL},
    {PA_14,NULL,NULL},
    {PA_15,NULL,NULL},
    {PA_16,NULL,NULL},
    {PA_17,NULL,NULL},
    {PA_18,NULL,NULL},
    {PA_19,NULL,NULL},
    {PA_20,NULL,NULL},
    {PA_21,NULL,NULL},
    {PA_22,NULL,NULL},
    {PA_23,NULL,NULL},
    {PA_24,NULL,NULL},
    {PA_25,NULL,NULL},
    {PA_26,NULL,NULL},
    {PA_27,NULL,NULL},
    {PA_28,NULL,NULL},
    {PA_29,NULL,NULL},
    {PA_30,NULL,NULL},
    {PA_31,NULL,NULL},
    {PB_0,NULL,NULL},
    {PB_1,NULL,NULL},
    {PB_2,NULL,NULL},
    {PB_3,NULL,NULL},
    {PB_4,NULL,NULL},
    {PB_5,NULL,NULL},
    {PB_6,NULL,NULL},
    {PB_7,NULL,NULL},
    {PB_8,NULL,NULL},
};

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief 用于设置gpio的输入输出
 * 
 * @param[in]       port     端口号
   @param[in]       in       输入/输出
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_inout_set(const TY_GPIO_PORT_E port, const bool in) 
{
    if(port >= CNTSOF(gpio_map_list)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    if(NULL == gpio_map_list[port].pf_obj) {
        gpio_map_list[port].pf_obj = tuya_hal_system_malloc(sizeof(gpio_t));
        if(NULL == gpio_map_list[port].pf_obj) {
            return OPRT_OS_ADAPTER_MALLOC_FAILED;
        }
    }
    
    if(TRUE == in) {
        gpio_init(gpio_map_list[port].pf_obj, gpio_map_list[port].pf_port);
        gpio_dir(gpio_map_list[port].pf_obj, PIN_INPUT);     // Direction: Input
        gpio_mode(gpio_map_list[port].pf_obj, PullUp);
    }else {
        gpio_init(gpio_map_list[port].pf_obj, gpio_map_list[port].pf_port);
        gpio_dir(gpio_map_list[port].pf_obj, PIN_OUTPUT);
        gpio_mode(gpio_map_list[port].pf_obj, PullNone);//PullUp
    }

    return OPRT_OS_ADAPTER_OK;
}


/**
 * @brief 用于设置gpio的上下拉模式
 * 
 * @param[in]       port     端口号
   @param[in]       mode     模式
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_mode_set(const TY_GPIO_PORT_E port, const TY_GPIO_MODE_E mode)
{
    if(port >= CNTSOF(gpio_map_list)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    switch (mode) {
        case TY_GPIO_PULLUP:
            gpio_mode(gpio_map_list[port].pf_obj, PullUp);
            break;
        case TY_GPIO_PULLDOWN:
            gpio_mode(gpio_map_list[port].pf_obj, PullDown);
            break;
        case TY_GPIO_PULLUP_PULLDOWN:
            // NOT SUPPORT
            return -1;
            break;
        case TY_GPIO_OPENDRAIN:
            gpio_mode(gpio_map_list[port].pf_obj, OpenDrain);
            break;
        case TY_GPIO_FLOATING:
            gpio_mode(gpio_map_list[port].pf_obj, PullNone);
            break;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 用于读取gpio的电平
 * 
 * @param[in]       port     端口号
 * @return gpio的电平值 0/1
 */
int tuya_hal_gpio_read(const TY_GPIO_PORT_E port)
{
    if(port >= CNTSOF(gpio_map_list)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    if(NULL == gpio_map_list[port].pf_obj) {
        return OPRT_OS_ADAPTER_COM_ERROR;
    }

    return gpio_read(gpio_map_list[port].pf_obj);

}

/**
 * @brief 用于设置gpio的电平
 * 
 * @param[in]       port     端口号
 * @param[in]       high     电平值
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_write(const TY_GPIO_PORT_E port, const bool high)
{
    if(port >= CNTSOF(gpio_map_list)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    if(NULL == gpio_map_list[port].pf_obj) {
        return OPRT_OS_ADAPTER_COM_ERROR;
    }

    gpio_write(gpio_map_list[port].pf_obj,high);

    return OPRT_OS_ADAPTER_OK;
}

void _gpio_irq_handler(uint32_t id, gpio_irq_event event)
{
    if(gpio_map_list[id].cb) {
        TY_GPIO_IRQ_EVENT e = event - 1;
        gpio_map_list[id].cb(id, e);
    }
}

/**
 * @brief 用于初始化gpio的中断模式
 * 
 * @param[in]       port     端口号
 * @param[in]       cb       中断回调
 * @param[in]       trig_type     中断触发条件
 * @return int 0=成功，非0=失败
 */
int tuya_hal_gpio_irq_init(const TY_GPIO_PORT_E port, const TY_GPIO_CALLBACK cb, const TY_GPIO_IRQ_TRIG_TYPE trig_type)
{
    if(port >= CNTSOF(gpio_map_list)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    if((trig_type < TY_IRQ_NONE) || (trig_type > TY_IRQ_LOW)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    if(trig_type == TY_IRQ_BOTH) { //rtk not support
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    gpio_irq_event trig = IRQ_NONE;
    if(trig_type == TY_IRQ_NONE) trig = IRQ_NONE;
    else if(trig_type == TY_IRQ_RISE) trig = IRQ_RISE;
    else if(trig_type == TY_IRQ_FALL) trig = IRQ_FALL;
    else if(trig_type == TY_IRQ_HIGH) trig = IRQ_HIGH;
    else if(trig_type == TY_IRQ_LOW) trig = IRQ_LOW;
 
    
    if(NULL == gpio_map_list[port].pf_obj) {
        gpio_map_list[port].pf_obj = tuya_hal_system_malloc(sizeof(gpio_t));
        if(NULL == gpio_map_list[port].pf_obj) {
            return OPRT_OS_ADAPTER_MALLOC_FAILED;
        }
    }
    
    gpio_irq_init((gpio_irq_t *)(gpio_map_list[port].pf_obj), gpio_map_list[port].pf_port, _gpio_irq_handler, port);
    
    if(trig != IRQ_NONE) {
        gpio_irq_set((gpio_irq_t *)(gpio_map_list[port].pf_obj), trig, TRUE);
        gpio_irq_enable((gpio_irq_t *)(gpio_map_list[port].pf_obj));
    }else {
        gpio_irq_set((gpio_irq_t *)(gpio_map_list[port].pf_obj), trig, FALSE);
        gpio_irq_disable((gpio_irq_t *)(gpio_map_list[port].pf_obj));       
    }

    if(gpio_map_list[port].cb == NULL) {
        gpio_map_list[port].cb = cb;
    }


    return OPRT_OS_ADAPTER_OK;
}





