/**
 * @file tuya_hal_i2c.h
 * @brief IIC总线操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */
#include "tuya_hal_i2c.h"
#include "tuya_os_adapter_errcode.h"
#include "i2c_api.h"
/***********************************************************
*************************micro define***********************
***********************************************************/
//I2C引脚选择 || 从机地址 || 频率定义
//I2C1_SEL S1 
#if 1
#define TUYA_I2C_MTR_SDA    PA_19
#define TUYA_I2C_MTR_SCL    PA_22
#else 
#define TUYA_I2C_MTR_SDA    PA_23
#define TUYA_I2C_MTR_SCL    PA_18
#endif
#define TUYA_I2C_BUS_CLK        100000 //100Khz 时钟频率
#define TUYA_I2C_PORT_COUNT     (1)

/***********************************************************
*************************variable define********************
***********************************************************/
static i2c_t g_i2c[TUYA_I2C_PORT_COUNT] = {};

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief 打开I2C设备
 * 
 * @param[in]   port    设备编号
 * 
 * @retval  =0      成功
 * @retval  <0      错误码
 */
int tuya_hal_i2c_open(const unsigned int port)
{
    if (port > TUYA_I2C_PORT_COUNT) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    i2c_init(&g_i2c[port], TUYA_I2C_MTR_SDA , TUYA_I2C_MTR_SCL);
    i2c_frequency(&g_i2c[port], TUYA_I2C_BUS_CLK);
    
    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 关闭I2C设备
 * 
 * @param[in]   port    设备编号
 * 
 * @retval  =0      成功
 * @retval  <0      错误码
 */
int tuya_hal_i2c_close(const unsigned int port)
{
    if (port > TUYA_I2C_PORT_COUNT) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 读数据
 * 
 * @param[in]   port    设备编号
 * @param[in]   addr    从机地址
 * @param[out]   buf     读取的数据缓冲区地址
 * @param[in]   len     缓冲区长度
 * 
 * @retval  >=0     实际读取的字节数
 * @retval  <0      错误码
 */
int tuya_hal_i2c_read(const unsigned int port, const unsigned int addr, unsigned char* buf, const unsigned int len)
{
    if (port > TUYA_I2C_PORT_COUNT) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    return i2c_read_timeout(&g_i2c[port], addr, buf, len, 1, 5);
}

/**
 * @brief 写数据
 * 
 * @param[in]   port    设备编号
 * @param[in]   addr    从机地址
 * @param[in]   buf     写入的数据缓冲区地址
 * @param[in]   len     缓冲区长度
 * 
 * @retval  >=0     实际写入的字节数
 * @retval  <0      错误码
 */
int tuya_hal_i2c_write(const unsigned int port, const unsigned int addr, const unsigned char* buf, const unsigned int len)
{
    if (port > TUYA_I2C_PORT_COUNT) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    return i2c_write_timeout(&g_i2c[port], addr, buf, len, 1, 5);
}

