/**
 * @file tuya_hal_ota.c
 * @brief ota底层操作接口
 * 
 * @copyright Copyright (c) {2018-2020} 涂鸦科技 www.tuya.com
 * 
 */
#include "tuya_hal_ota.h"
#include "tuya_os_adapter.h"
#include "tuya_os_adapter_errcode.h"
#include "tuya_hal_system.h"
#include "flash_api.h"
#include <device_lock.h>
#include "rtl8710b_ota.h"

/***********************************************************
*************************micro define***********************
***********************************************************/
typedef enum {
    UGS_RECV_HEADER = 0,
    UGS_RECV_OTA_HD,
    UGS_SEARCH_SIGN,
    UGS_RECV_SIGN,
    UGS_RECV_IMG_DATA,
    UGS_FINISH
} UG_STAT_E;

typedef struct {
    UG_STAT_E stat;
    update_ota_target_hdr hdr;
    unsigned int ota_index;

    unsigned int image_cnt;
    unsigned char signature[8];
    update_dw_info DownloadInfo[2];
    unsigned int cur_image_cnt[2];

    flash_t flash;
    unsigned int new_addr;
    unsigned int recv_data_cnt;
} UG_PROC_S;

/***********************************************************
*************************variable define********************
***********************************************************/
static UG_PROC_S *ug_proc = NULL;

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief 升级开始通知函数
 * 
 * @param[in] file_size 升级固件大小
 * 
 * @retval  =0      成功
 * @retval  <0      错误码
 */
int tuya_hal_ota_start_inform(unsigned int file_size)
{
    if(file_size == 0) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    if(ug_proc == NULL) {
        ug_proc = tuya_hal_system_malloc(sizeof(UG_PROC_S));
        if(NULL == ug_proc) {
            return OPRT_OS_ADAPTER_MALLOC_FAILED;
        }
    }
    
    memset(ug_proc,0,sizeof(UG_PROC_S));

    if (ota_get_cur_index() == OTA_INDEX_1) {
        ug_proc->ota_index = OTA_INDEX_2;
        tuya_hal_output_log("OTA2 address space will be upgraded\r\n");
    } else {
        ug_proc->ota_index = OTA_INDEX_1;
        tuya_hal_output_log("OTA1 address space will be upgraded\r\n");
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief ota数据包处理
 * 
 * @param[in] total_len ota升级包总大小
 * @param[in] offset 当前data在升级包中的偏移
 * @param[in] data ota数据buffer指针
 * @param[in] len ota数据buffer长度
 * @param[out] remain_len 内部已经下发但该函数还未处理的数据长度
 * @param[in] pri_data 保留参数
 *
 * @retval  =0      成功
 * @retval  <0      错误码
 */
int tuya_hal_ota_data_process(const unsigned int total_len, const unsigned int offset,
                              const unsigned char* data, const unsigned int len, unsigned int* remain_len, void* pri_data)
{
    static unsigned int ota_hd_len = 0;

    switch(ug_proc->stat) {
        case UGS_RECV_HEADER: {
            if(len < sizeof(update_file_hdr)) {
                *remain_len = len;
                break;
            }

            memcpy(&ug_proc->hdr.FileHdr,data,sizeof(update_file_hdr));
            ug_proc->stat = UGS_RECV_OTA_HD;
            ota_hd_len = (ug_proc->hdr.FileHdr.HdrNum) * sizeof(update_file_img_hdr);
            *remain_len = len;
        }
        //break;

        case UGS_RECV_OTA_HD: {
            if(len < ota_hd_len+sizeof(update_file_hdr)) {
                *remain_len = len;
                break;
            }

            char * pImgId = NULL;
            if(OTA_INDEX_1 == ug_proc->ota_index) {
                pImgId = "OTA1";
            }else {
                pImgId = "OTA2";
            }

            unsigned int ret = 0;
            ret = get_ota_tartget_header(data,(ota_hd_len+sizeof(update_file_hdr)), \
                                         &(ug_proc->hdr), pImgId);
            if(0 == ret) {
                tuya_hal_output_log("get_ota_tartget_header err\r\n");
                return OPRT_OS_ADAPTER_OTA_PROCESS_FAILED;
            }

            // get new image addr and check new address validity
            if(!get_ota_address(ug_proc->ota_index, &ug_proc->new_addr, &(ug_proc->hdr))) {
                tuya_hal_output_log("get_ota_address err\r\n");
                return OPRT_OS_ADAPTER_OTA_PROCESS_FAILED;
            }

            unsigned int new_img_len = ug_proc->hdr.FileImgHdr.ImgLen;
        #if OPERATING_SYSTEM == SYSTEM_REALTEK8710_1M
            if(new_img_len > (468*1024 -32)) {//32 Bytes resvd for header 468K
                tuya_hal_output_log("image length is too big\r\n");
                return OPRT_OS_ADAPTER_OTA_PROCESS_FAILED;
            }
        #endif

        #if OPERATING_SYSTEM == SYSTEM_REALTEK8710_2M
            if(new_img_len > (788*1024 - 32)) {//32 Bytes resvd for header 960K
                tuya_hal_output_log("image length is too big\r\n");
                return OPRT_OS_ADAPTER_OTA_PROCESS_FAILED;
            }
        #endif
            
            erase_ota_target_flash(ug_proc->new_addr, new_img_len);
            if(ug_proc->hdr.RdpStatus == ENABLE) {
                device_mutex_lock(RT_DEV_LOCK_FLASH);
                flash_erase_sector(&ug_proc->flash, RDP_FLASH_ADDR - SPI_FLASH_BASE);
                device_mutex_unlock(RT_DEV_LOCK_FLASH);
            }

            // 暂时仅支持image的升级
            if(ug_proc->hdr.RdpStatus == ENABLE) {
                ug_proc->image_cnt = 2;
                if(ug_proc->hdr.FileImgHdr.Offset < ug_proc->hdr.FileRdpHdr.Offset) {
                    ug_proc->DownloadInfo[0].ImgId = OTA_IMAG;
                    /* get OTA image and Write New Image to flash, skip the signature,
                    not write signature first for power down protection*/
                    ug_proc->DownloadInfo[0].FlashAddr = ug_proc->new_addr -SPI_FLASH_BASE + 8;
                    ug_proc->DownloadInfo[0].ImageLen = ug_proc->hdr.FileImgHdr.ImgLen - 8;/*skip the signature*/
                    ug_proc->DownloadInfo[0].ImgOffset = ug_proc->hdr.FileImgHdr.Offset;
                } else {
                    ug_proc->DownloadInfo[0].ImgId = OTA_IMAG;
                    /* get OTA image and Write New Image to flash, skip the signature,
                    not write signature first for power down protection*/
                    ug_proc->DownloadInfo[0].FlashAddr = ug_proc->new_addr -SPI_FLASH_BASE + 8;
                    ug_proc->DownloadInfo[0].ImageLen = ug_proc->hdr.FileImgHdr.ImgLen - 8;/*skip the signature*/
                    ug_proc->DownloadInfo[0].ImgOffset = ug_proc->hdr.FileImgHdr.Offset;
                }
            }else {
                ug_proc->image_cnt = 1;
                ug_proc->DownloadInfo[0].ImgId = OTA_IMAG;
                /* get OTA image and Write New Image to flash, skip the signature,
                not write signature first for power down protection*/
                ug_proc->DownloadInfo[0].FlashAddr = ug_proc->new_addr -SPI_FLASH_BASE + 8;
                ug_proc->DownloadInfo[0].ImageLen = ug_proc->hdr.FileImgHdr.ImgLen - 8;/*skip the signature*/
                ug_proc->DownloadInfo[0].ImgOffset = ug_proc->hdr.FileImgHdr.Offset;
            }
            
            if(ug_proc->hdr.RdpStatus == ENABLE) {
                //tuya_hal_output_log("RDP Image Address = %x\r\n", RDP_FLASH_ADDR);
            }

            ug_proc->recv_data_cnt = ota_hd_len + sizeof(update_file_hdr);
            *remain_len = len - (ota_hd_len+sizeof(update_file_hdr));
            ug_proc->stat = UGS_SEARCH_SIGN;
        }
        break;

        case UGS_SEARCH_SIGN: {
            if(ug_proc->recv_data_cnt + len < ug_proc->DownloadInfo[0].ImgOffset) {
                ug_proc->recv_data_cnt += len;
                *remain_len = 0;
                break;
            }

            int offset = (ug_proc->DownloadInfo[0].ImgOffset - ug_proc->recv_data_cnt);
            *remain_len = len - offset;
            if(*remain_len < sizeof(ug_proc->signature)) {
                ug_proc->recv_data_cnt += offset;
                ug_proc->stat = UGS_RECV_SIGN;
            }else {
                mempcpy(ug_proc->signature,data+offset,sizeof(ug_proc->signature));
                *remain_len -= sizeof(ug_proc->signature);
                ug_proc->stat = UGS_RECV_IMG_DATA;
            }
        }
        break;

        case UGS_RECV_SIGN: {
            if(len < sizeof(ug_proc->signature)) { // 8 is signature
                *remain_len = len;
                break;
            }

            mempcpy(ug_proc->signature,data,sizeof(ug_proc->signature));
            ug_proc->recv_data_cnt += 8;
            *remain_len = len-8;
            ug_proc->stat = UGS_RECV_IMG_DATA;
        }
        break;

        case UGS_RECV_IMG_DATA: {
        #define RT_IMG_WR_UNIT 1024
            if(ug_proc->cur_image_cnt[0] >= ug_proc->DownloadInfo[0].ImageLen) {
                ug_proc->stat = UGS_FINISH;
                *remain_len = len;
                break;
            }

            if((ug_proc->cur_image_cnt[0] + len < ug_proc->DownloadInfo[0].ImageLen) && \
               (len < RT_IMG_WR_UNIT)) {
                *remain_len = len;
                break;
            }

            unsigned int write_len = 0;
            if(ug_proc->cur_image_cnt[0] + len < ug_proc->DownloadInfo[0].ImageLen) {
                write_len = RT_IMG_WR_UNIT;
            }else {
                write_len = (ug_proc->DownloadInfo[0].ImageLen - ug_proc->cur_image_cnt[0]);
            }
            
            device_mutex_lock(RT_DEV_LOCK_FLASH);
            if(flash_stream_write(&ug_proc->flash, ug_proc->DownloadInfo[0].FlashAddr + ug_proc->cur_image_cnt[0], \
                                  write_len, data) < 0) {
                device_mutex_unlock(RT_DEV_LOCK_FLASH);
                tuya_hal_output_log("Write sector failed\r\n");
                return OPRT_OS_ADAPTER_OTA_PROCESS_FAILED;
            }
            device_mutex_unlock(RT_DEV_LOCK_FLASH);
            
            ug_proc->cur_image_cnt[0] += write_len;
            ug_proc->recv_data_cnt += write_len;
            *remain_len = len - write_len;

            if(ug_proc->cur_image_cnt[0] >= ug_proc->DownloadInfo[0].ImageLen) {
                ug_proc->stat = UGS_FINISH;
                break;
            }
        }
        break;

        case UGS_FINISH: {
            *remain_len = 0;
        }
        break;
    }
    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 固件ota数据传输完毕通知
 *        用户可以做固件校验以及设备重启
 * param[in]        reset       是否需要重启
 * @retval  =0      成功
 * @retval  <0      错误码
 */
int tuya_hal_ota_end_inform(bool reset)
{
    // verify
    unsigned int ret = 0;
    ret = verify_ota_checksum(ug_proc->new_addr,ug_proc->DownloadInfo[0].ImageLen, \
                              ug_proc->signature,&ug_proc->hdr);
    if(1 != ret) {
        tuya_hal_output_log("verify_ota_checksum err\r\n");
        return OPRT_OS_ADAPTER_OTA_VERIFY_FAILED;
    }

    if(!change_ota_signature(ug_proc->new_addr, ug_proc->signature, ug_proc->ota_index)) {
        tuya_hal_output_log("change signature failed\r\n");
        return OPRT_OS_ADAPTER_OTA_END_INFORM_FAILED;
    }

    tuya_hal_output_log("the gateway upgrade success,now go to reset!\r\n");

    tuya_hal_system_free(ug_proc);
    ug_proc = NULL;

    if(reset == TRUE) {
        ota_platform_reset();
    }
    
    return OPRT_OS_ADAPTER_OK;
}

