/**
 * @file tuya_hal_storge.c
 * @brief flash操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#define __UNI_STORGE_GLOBALS
#include "flash_api.h"
#include "rtl8710b.h"
#include <device_lock.h>
#include <basic_types.h>
#include "tuya_hal_storge.h"
#include "tuya_os_adapter_errcode.h"

/***********************************************************
*************************micro define***********************
***********************************************************/
#define PARTITION_SIZE (1 << 12) /* 4KB */
#define FLH_BLOCK_SZ PARTITION_SIZE

// flash map   0x000F5000  0x000FD000   0x8000
#define SIMPLE_FLASH_START (0x200000 - 0x10000 - 0x4000)
#define SIMPLE_FLASH_SIZE 0x10000 // 64k
#define SIMPLE_FLASH_SWAP_START (0x200000 - 0x4000)
#define SIMPLE_FLASH_SWAP_SIZE 0x4000 // 16k
#define SIMPLE_FLASH_KEY_ADDR  (0x200000 - 0x10000 - 0x4000 - 0x1000)


/***********************************************************
*************************variable define********************
***********************************************************/
static flash_t obj;

static UNI_STORAGE_DESC_S storage = {
    SIMPLE_FLASH_START,
    SIMPLE_FLASH_SIZE,
    FLH_BLOCK_SZ,
    SIMPLE_FLASH_SWAP_START,
    SIMPLE_FLASH_SWAP_SIZE,
    SIMPLE_FLASH_KEY_ADDR
};

static UF_PARTITION_TABLE_S uf_file = {
    .sector_size = PARTITION_SIZE,
    .uf_partition_num = 2,
    .uf_partition = {
        {SIMPLE_FLASH_KEY_ADDR - 0x8000, 0x8000},
        {SIMPLE_FLASH_KEY_ADDR - 0x8000 - 0x18000, 0x18000}
    }
};

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief read data from flash
 * 
 * @param[in]       addr        flash address
 * @param[out]      dst         pointer of buffer
 * @param[in]       size        size of buffer
 * @return int 0=成功，非0=失败
 */
int tuya_hal_flash_read(const unsigned int addr, unsigned char *dst, const unsigned int size)
{
    if(NULL == dst) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    device_mutex_lock(RT_DEV_LOCK_FLASH);
    flash_stream_read(&obj, addr, size, dst);
    device_mutex_unlock(RT_DEV_LOCK_FLASH);

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief write data to flash
 * 
 * @param[in]       addr        flash address
 * @param[in]       src         pointer of buffer
 * @param[in]       size        size of buffer
 * @return int 0=成功，非0=失败
 */
int tuya_hal_flash_write(const unsigned int addr, const unsigned char *src, const unsigned int size)
{
    if(NULL == src) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    device_mutex_lock(RT_DEV_LOCK_FLASH);
    flash_stream_write(&obj, addr, size, src);
    device_mutex_unlock(RT_DEV_LOCK_FLASH);

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief erase flash block
 * 
 * @param[in]       addr        flash block address
 * @param[in]       size        size of flash block
 * @return int 0=成功，非0=失败
 */
int tuya_hal_flash_erase(const unsigned int addr, const unsigned int size)
{
    uint16_t start_sec = (addr/PARTITION_SIZE);
    uint16_t end_sec = ((addr+size-1)/PARTITION_SIZE);

    int i = 0;
    for(i = start_sec;i <= end_sec;i++) {
        device_mutex_lock(RT_DEV_LOCK_FLASH);
        flash_erase_sector(&obj, PARTITION_SIZE*i);
        device_mutex_unlock(RT_DEV_LOCK_FLASH);
    }

    return OPRT_OS_ADAPTER_OK;
}


/**
 * @brief get description of storage
 * 
 * @return  pointer to storage description structure
 */
UNI_STORAGE_DESC_S *tuya_hal_storage_get_desc(VOID)
{
    return &storage;
}


/**
 * @brief get UF file description
 * 
 * @return  pointer to descrtion of UF file
 */
UF_PARTITION_TABLE_S *tuya_hal_uf_get_desc(VOID)
{
    return &uf_file;
}

