/**
 * @file tuya_hal_uart.c
 * @brief UART操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#include "objects.h"
#include "serial_api.h"
#include "freertos_pmu.h"
#include "tuya_hal_uart.h"
#include "tuya_os_adapter_errcode.h"
#include "tuya_hal_semaphore.h"

/***********************************************************
*************************micro define***********************
***********************************************************/
#define UART_TX PA_23 //UART0 TX
#define UART_RX PA_18 //UART0 RX 
#define UART_RTS PA_22
#define UART_CTS PA_19

/***********************************************************
*************************variable define********************
***********************************************************/
extern void set_uart_lp(int port,bool enable);
serial_t sobj;
UART_INT_RX_CB uart0_int_rx_cb = NULL;

/***********************************************************
*************************function define********************
***********************************************************/
static void __uart_irq(unsigned int id, SerialIrq event)
{
    serial_t *xsobj = (void*)id;
    if(xsobj != &sobj) {
        return;
    }
    
    if(event == RxIrq) {
        unsigned char ch = serial_getc(xsobj);
        if(uart0_int_rx_cb)
            uart0_int_rx_cb(TY_UART1, ch);
    }
}

static u32 _uart0_suspend(u32 expected_idle_time )
{
    return TRUE;
}

static u32 _uart0_resume(u32 expected_idle_time)
{
    NVIC_SetPriority(UART0_IRQ, 10);
    NVIC_EnableIRQ(UART0_IRQ);
    return TRUE;
}

/**
 * @brief 用于初始化串口
 * 
 * @param[in]  uart     串口配置信息
 */
int tuya_hal_uart_init(const TY_UART_DEV_S *uart)
{
    if(uart == NULL) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    if(uart->port != TY_UART1) {  //rtk  uart0 is user uart port
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    TY_UART_PORT_E port = uart->port;
    if(uart->port == TY_UART1) {
        port = TY_UART0;
    }
    
    int data_bit = 0;
    if(uart->cfg.data_bits == TY_UART_DATA_BIT5) {
        data_bit = 5;
    }else if(uart->cfg.data_bits == TY_UART_DATA_BIT6) {
        data_bit = 6;
    }else if(uart->cfg.data_bits == TY_UART_DATA_BIT7) {
        data_bit = 7;
    }else if(uart->cfg.data_bits == TY_UART_DATA_BIT8) {
        data_bit = 8;
    }else {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }
    
    int stop_bit = uart->cfg.stop_bits;
    if((uart->cfg.stop_bits < TY_UART_STOP_BIT1) || \
       (uart->cfg.stop_bits > TY_UART_STOP_BIT2)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    int parity_bit = uart->cfg.parity;
    if((uart->cfg.parity < TY_UART_PARITY_NONE) || \
       (uart->cfg.parity > TY_UART_PARITY_EVEN)) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    set_uart_lp(port,TRUE);//支持tickless模式
    serial_init(&sobj, UART_TX,UART_RX);
    serial_baud(&sobj, uart->cfg.baudrate);
    serial_format(&sobj,data_bit,parity_bit,stop_bit);

    if(uart->cfg.intrx == TY_UART_INT_RX_ENABLE) {
        serial_irq_handler(&sobj, __uart_irq, (uint32_t)&sobj);
        serial_irq_set(&sobj, RxIrq, 1);
        uart0_int_rx_cb = uart->rx_cb;
    }

    if(uart->cfg.flowctl == TY_UART_FLOW_CONTROL_ENABLE) {
        pin_mode(UART_CTS, PullDown); //init CTS in low
        serial_set_flow_control(&sobj, FlowControlRTSCTS, UART_RTS, UART_CTS);
    }

    //支持tickless模式
    pmu_register_sleep_callback(PMU_UART0_DEVICE, (PSM_HOOK_FUN)_uart0_suspend, (void*)NULL, (PSM_HOOK_FUN)_uart0_resume, (void*)NULL);


    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 用于释放串口
 * 
 * @param[in]  port     串口id
 */
int tuya_hal_uart_deinit(const TY_UART_PORT_E port)
{
    if(port != TY_UART1) { //rtl8710bn only support uart0 for user
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    
    pmu_unregister_sleep_callback(PMU_UART0_DEVICE);
    serial_free(&sobj);
    
    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief 用于发送一个字节
 * 
 * @param[in]  port     串口id
 * @param[ch]  ch       要发送的数据
 */
int tuya_hal_uart_send_char(const TY_UART_PORT_E port, const unsigned char ch)
{
    if(port != TY_UART1) { //rtl8710bn only support uart0 for user
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    serial_putc(&sobj, ch);

    return OPRT_OS_ADAPTER_OK;
}

