/**
 * @file tuya_hal_semaphore.c
 * @brief semaphore相关接口封装
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#define _UNI_SEMAPHORE_GLOBAL
#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"
#include "basic_types.h"
#include "tuya_hal_semaphore.h"
#include "tuya_os_adapter_errcode.h"

/***********************************************************
*************************micro define***********************
***********************************************************/
typedef struct
{
    xSemaphoreHandle sem;
}SEM_MANAGE, *P_SEM_MANAGE;


/***********************************************************
*************************variable define********************
***********************************************************/

/***********************************************************
*************************function define********************
***********************************************************/
static SEM_HANDLE CreateSemaphore(VOID)
{
    P_SEM_MANAGE pSemManage;
    
    pSemManage = (P_SEM_MANAGE)tuya_hal_system_malloc(sizeof(SEM_MANAGE));

    return (SEM_HANDLE)pSemManage;
}


static int InitSemaphore(const SEM_HANDLE semHandle, const uint32_t semCnt,\
                         const uint32_t sem_max)
{
    if(!semHandle)
        return OPRT_OS_ADAPTER_INVALID_PARM;
        
    P_SEM_MANAGE pSemManage;
    pSemManage = (P_SEM_MANAGE)semHandle;

    pSemManage->sem = xSemaphoreCreateCounting(sem_max, semCnt);
    if(NULL == pSemManage->sem) {
        return OPRT_OS_ADAPTER_SEM_CREAT_FAILED;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief tuya_hal_semaphore_create_init用于创建并初始化semaphore
 * 
 * @param[out] *pHandle semaphore句柄
 * @param[in] semCnt 
 * @param[in] sem_max 
 * @return int 0=成功，非0=失败
*/
int tuya_hal_semaphore_create_init(SEM_HANDLE *pHandle, const unsigned int semCnt, \
                                   const unsigned int sem_max)
{
    if(NULL == pHandle) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    *pHandle = CreateSemaphore();
    if(*pHandle == NULL) {
        return OPRT_OS_ADAPTER_MALLOC_FAILED;
    }

    int ret = InitSemaphore(*pHandle, semCnt, sem_max);
    if(ret != OPRT_OS_ADAPTER_OK) {
        tuya_hal_system_free(*pHandle);
        *pHandle = NULL;
        return ret;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief tuya_hal_semaphore_wait用于wait semaphore
 * 
 * @param[in] semHandle semaphore句柄
 * @return int 0=成功，非0=失败
*/
int tuya_hal_semaphore_wait(const SEM_HANDLE semHandle)
{
    if(!semHandle) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    P_SEM_MANAGE pSemManage;
    pSemManage = (P_SEM_MANAGE)semHandle;

    BaseType_t ret;
    ret = xSemaphoreTake(pSemManage->sem, portMAX_DELAY);
    if(pdTRUE != ret) {
        return OPRT_OS_ADAPTER_SEM_WAIT_FAILED;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief tuya_hal_semaphore_post用于post semaphore
 * 
 * @param[in] semHandle semaphore句柄
 * @return int 0=成功，非0=失败
 */
int tuya_hal_semaphore_post(const SEM_HANDLE semHandle)
{
    if(!semHandle) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    P_SEM_MANAGE pSemManage;
    pSemManage = (P_SEM_MANAGE)semHandle;

    BaseType_t ret;
    if(FALSE == tuya_hal_system_isrstatus()) {
        ret = xSemaphoreGive(pSemManage->sem);
    }else {
        signed portBASE_TYPE xHigherPriorityTaskWoken = pdFALSE;
        ret = xSemaphoreGiveFromISR(pSemManage->sem,
                                    &xHigherPriorityTaskWoken);
        portEND_SWITCHING_ISR(xHigherPriorityTaskWoken);
    }

    if(pdTRUE != ret) {
        return OPRT_OS_ADAPTER_SEM_POST_FAILED;
    }

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief tuya_hal_semaphore_release用于release semaphore
 * 
 * @param[in] semHandle 
 * @return int 0=成功，非0=失败
 */
int tuya_hal_semaphore_release(const SEM_HANDLE semHandle)
{
    if(!semHandle) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    P_SEM_MANAGE pSemManage;
    pSemManage = (P_SEM_MANAGE)semHandle;

    vSemaphoreDelete(pSemManage->sem);
    tuya_hal_system_free(semHandle); // 释放信号量管理结构

    return OPRT_OS_ADAPTER_OK;
}


