/**
 * @file tuya_hal_system.c
 * @brief 操作系统相关接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */
#define _UNI_SYSTEM_GLOBAL
#include "FreeRTOS.h"
#include "basic_types.h"
#include "rtl8710b.h"
#include "rtl8710b_backup_reg.h"
#include "freertos_pmu.h"
#include "tuya_hal_system.h"
#include "tuya_os_adapter_errcode.h"

/***********************************************************
*************************micro define***********************
***********************************************************/

/***********************************************************
*************************variable define********************
***********************************************************/

/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief tuya_hal_get_systemtickcount用于获取系统运行ticket
 * @return SYS_TICK_T
 */
SYS_TICK_T tuya_hal_get_systemtickcount(void)
{
    return (SYS_TICK_T)xTaskGetTickCount();
}


/**
 * @brief tuya_hal_get_tickratems用于获取系统ticket是多少个ms
 * 
 * @return the time is ms of a system ticket
 */
unsigned int tuya_hal_get_tickratems(void)
{
    return (unsigned int)portTICK_RATE_MS;
}


/**
 * @brief tuya_hal_system_sleep用于系统sleep
 * 
 * @param[in] msTime sleep time is ms
 */
void tuya_hal_system_sleep(const unsigned long msTime)
{
    vTaskDelay((msTime)/(portTICK_RATE_MS));
}


/**
 * @brief tuya_hal_system_isrstatus用于检查系统是否处于中断上下文
 * 
 * @return true 
 * @return false 
 */
bool tuya_hal_system_isrstatus(void)
{
    if(0 != __get_IPSR()) {
        return TRUE;
    }

    return FALSE;
}


/**
 * @brief tuya_hal_system_reset用于重启系统
 * 
 */
void tuya_hal_system_reset(void)
{
    sys_reset();
}


/**
 * @brief tuya_hal_system_getheapsize用于获取堆大小/剩余内存大小
 * 
 * @return int <0: don't support  >=0: current heap size/free memory
 */
int tuya_hal_system_getheapsize(void)
{
    return (int)xPortGetFreeHeapSize();
}


/**
 * @brief tuya_hal_system_get_rst_info用于获取硬件重启原因
 * 
 * @return 设备重启原因
 */
TY_RST_REASON_E tuya_hal_system_get_rst_info(void)
{
    int value_bk0 = BKUP_Read(BKUP_REG0);

    unsigned char value = value_bk0 & 0xFF;
    TY_RST_REASON_E rtn_value;
    switch(value) {
        case 0: {//硬件重启
             rtn_value = TY_RST_POWER_OFF;
         } 
         break;
         
        case 1: {//看门狗复位
            rtn_value = TY_RST_SOFTWARE_WATCHDOG;
            BKUP_Clear(BKUP_REG0, BIT_CPU_RESET_HAPPEN);
        } 
        break;
        
        case 5: {//软件重启
            rtn_value = TY_RST_SOFTWARE;
            BKUP_Clear(BKUP_REG0, BIT_CPU_RESET_HAPPEN);
            BKUP_Clear(BKUP_REG0, BIT_RTC_RESTORE);
        } 
        break;
        
        default: {//其它
            rtn_value = TY_RST_OTHER;
            BKUP_Clear(BKUP_REG0, 0x01);
            BKUP_Clear(BKUP_REG0, 0x02);
            BKUP_Clear(BKUP_REG0, 0x04);
            BKUP_Clear(BKUP_REG0, 0x08);
            BKUP_Clear(BKUP_REG0, 0x10);
            BKUP_Clear(BKUP_REG0, 0x20);
            BKUP_Clear(BKUP_REG0, 0x40);
            BKUP_Clear(BKUP_REG0, 0x80);
        } 
        break;
    }
    
    return rtn_value;
}


/**
 * @brief tuya_hal_get_random_data用于获取指定条件下的随机数
 * 
 * @param[in] range 
 * @return 随机数
 */
int tuya_hal_get_random_data(const unsigned int range)
{
    unsigned int trange = range;
    if(range == 0)
        trange = 0xFF;

    static unsigned char exec_flag = FALSE;
    extern random_seed;
    if(!exec_flag) {
        srand(random_seed);
        exec_flag = TRUE;
    }
    
    unsigned int val =  rand() + random_seed;
    return (val % trange);
}



/**
 * @brief tuya_hal_set_cpu_lp_mode用于设置cpu的低功耗模式
 * 
 * @param[in] en 
 * @param[in] mode
 * @return int 0=成功，非0=失败
 */
unsigned char cpu_lp_flag = 0;
int tuya_hal_set_cpu_lp_mode(const bool en,const TY_CPU_SLEEP_MODE_E mode)
{
    if(mode == TY_CPU_SLEEP) {
        if(en) {
            if(cpu_lp_flag == 0) {
                cpu_lp_flag = 1;
                pmu_release_wakelock(PMU_OS);
            }

            pmu_release_wakelock(PMU_DEV_USER_BASE);
        }else {
            pmu_acquire_wakelock(PMU_DEV_USER_BASE);
        }
    }else {
        return OPRT_OS_ADAPTER_CPU_LPMODE_SET_FAILED;
    }

    return OPRT_OS_ADAPTER_OK;
}
