/**
 * @file tuya_hal_thread.c
 * @brief 线程操作接口
 * 
 * @copyright Copyright(C),2018-2020, 涂鸦科技 www.tuya.com
 * 
 */

#include <FreeRTOS.h>
#include <task.h>
#include "tuya_hal_thread.h"
#include "tuya_os_adapter_errcode.h"

/***********************************************************
*************************micro define***********************
***********************************************************/

/***********************************************************
*************************variable define********************
***********************************************************/


/***********************************************************
*************************function define********************
***********************************************************/
/**
 * @brief create and start a tuya sdk thread
 * 
 * @param[out] thread       the thread handle
 * @param[in] name          the thread name
 * @param[in] stack_size    stack size of thread
 * @param[in] priority      priority of thread
 * @param[in] func          the main thread process function
 * @param[in] arg           the args of the func, can be null
 * @retval OPRT_OS_ADAPTER_OK          success
 * @retval Other            fail
 */
int tuya_hal_thread_create(THREAD_HANDLE* thread,
                           const char* name,
                           const unsigned int stack_size,
                           const unsigned int priority,
                           const THREAD_FUNC_T func,
                           const void* arg)
{
    BaseType_t ret = 0;
    ret = xTaskCreate(func, name, stack_size/sizeof(portSTACK_TYPE), arg, priority, thread);
    if (ret != pdPASS) {
        return OPRT_OS_ADAPTER_THRD_CREAT_FAILED;
    }

    return OPRT_OS_ADAPTER_OK;
}


/**
* @brief terminal thread and release thread resources
* 
* @param[in] thread    the input thread handle
* @retval OPRT_OS_ADAPTER_OK      success
* @retval Other        fail
*/
int tuya_hal_thread_release(THREAD_HANDLE thread)
{
//    if (NULL == thread) {
//        return OPRT_OS_ADAPTER_INVALID_PARM;
//    }

    // delete thread process 
    vTaskDelete(thread);

    return OPRT_OS_ADAPTER_OK;
}

/**
 * @brief check thread is self thread
 * 
 * @param[in] thread    the thread handle
 * @param[out] is_self  output is self thread
 * @retval OPRT_OS_ADAPTER_OK      success
 * @retval Other        fail
 */
int tuya_hal_thread_is_self(THREAD_HANDLE thread, bool* is_self)
{
    if (NULL == thread || NULL == is_self) {
        return OPRT_OS_ADAPTER_INVALID_PARM;
    }

    THREAD_HANDLE self_handle = xTaskGetCurrentTaskHandle();
    if (NULL == self_handle) {
        return OPRT_OS_ADAPTER_THRD_JUDGE_SELF_FAILED;
    }

    *is_self = (thread == self_handle);

    return OPRT_OS_ADAPTER_OK;
}
